<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * YQS SaaS Announcements Widget (server-side fetch + same-origin HTML endpoint)
 *
 * - Fetches latest announcements from SaaS platform REST.
 * - Caches results via transient.
 * - Exposes an admin-ajax endpoint returning HTML, suitable for async rendering.
 *
 * This file is designed to be copy-pasted into other plugins.
 */
class YQS_SaaS_Announcements_Widget {
	private static $instance = null;
	private $args = array();

	public static function instance( array $args = array() ) {
		if ( null === self::$instance ) {
			self::$instance = new self( $args );
		}
		return self::$instance;
	}

	private function __construct( array $args = array() ) {
		$defaults = array(
			'ajax_action'     => 'yqs_saas_announcements_widget',
			'capability'     => 'manage_options',
			'nonce_action'   => 'wp_aiw_nonce',
			'remote_endpoint' => 'https://youquso.com/wp-json/yqs/v1/announcements/latest?limit=5',
			'limit'          => 5,
			'ttl'            => 10 * MINUTE_IN_SECONDS,
			'cache_prefix'   => 'yqs_saas_ann_widget_',
			'user_agent'     => 'YQS-SaaS-Announcements-Widget/1.0',
		);
		$this->args = array_merge( $defaults, $args );

		$limit = isset( $this->args['limit'] ) ? (int) $this->args['limit'] : 5;
		$this->args['limit'] = max( 1, min( 10, $limit ) );

		$ttl = isset( $this->args['ttl'] ) ? (int) $this->args['ttl'] : ( 10 * MINUTE_IN_SECONDS );
		$this->args['ttl'] = max( 0, $ttl );

		$this->args['remote_endpoint'] = is_string( $this->args['remote_endpoint'] ) ? trim( $this->args['remote_endpoint'] ) : '';
		if ( $this->args['remote_endpoint'] === '' ) {
			$this->args['remote_endpoint'] = $defaults['remote_endpoint'];
		}

		add_action( 'wp_ajax_' . $this->args['ajax_action'], array( $this, 'ajax_render' ) );
	}

	public function ajax_render() {
		if ( ! current_user_can( $this->args['capability'] ) ) {
			wp_die( '无权限', '', array( 'response' => 403 ) );
		}

		check_ajax_referer( $this->args['nonce_action'], 'nonce' );

		$limit = isset( $_POST['limit'] ) ? absint( wp_unslash( $_POST['limit'] ) ) : (int) $this->args['limit'];
		$limit = max( 1, min( 10, $limit ) );

		$html = $this->render_widget_html( $limit );
		nocache_headers();
		header( 'Content-Type: text/html; charset=' . get_bloginfo( 'charset' ) );
		echo $html;
		wp_die();
	}

	public function render_widget_html( $limit = 5 ) {
		$items = $this->fetch_latest( (int) $limit );

		ob_start();
		if ( is_array( $items ) && ! empty( $items ) ) {
			echo '<ul class="wp-aiw-ann-list">';
			foreach ( $items as $it ) {
				$title = isset( $it['title'] ) ? (string) $it['title'] : '';
				$link  = isset( $it['link'] ) ? (string) $it['link'] : '';
				if ( $title === '' || $link === '' ) {
					continue;
				}
				echo '<li><a href="' . esc_url( $link ) . '" target="_blank" rel="noopener noreferrer">' . esc_html( $title ) . '</a></li>';
			}
			echo '</ul>';
		} else {
			echo '<p class="description" style="margin:0;">暂无更新公告</p>';
		}
		return (string) ob_get_clean();
	}

	private function fetch_latest( $limit ) {
		$limit = max( 1, min( 10, (int) $limit ) );
		$ttl = (int) apply_filters( 'yqs_saas_announcements_widget_ttl', $this->args['ttl'], $limit, $this->args );
		$ttl = max( 0, $ttl );

		$cache_key = $this->args['cache_prefix'] . md5( $this->args['remote_endpoint'] ) . '_' . $limit;
		if ( $ttl > 0 ) {
			$cached = get_transient( $cache_key );
			if ( is_array( $cached ) ) {
				return $cached;
			}
		}

		$url = add_query_arg( array( 'limit' => $limit ), $this->args['remote_endpoint'] );
		$args = array(
			'timeout' => 8,
			'headers' => array(
				'Accept' => 'application/json',
				'User-Agent' => (string) apply_filters( 'yqs_saas_announcements_widget_user_agent', $this->args['user_agent'], $this->args ),
			),
		);

		$resp = wp_remote_get( $url, $args );
		if ( is_wp_error( $resp ) ) {
			return array();
		}

		$code = (int) wp_remote_retrieve_response_code( $resp );
		$body = (string) wp_remote_retrieve_body( $resp );
		if ( $code < 200 || $code >= 300 || $body === '' ) {
			return array();
		}

		$data = json_decode( $body, true );
		if ( ! is_array( $data ) || empty( $data['items'] ) || ! is_array( $data['items'] ) ) {
			return array();
		}

		$items = array();
		foreach ( $data['items'] as $it ) {
			if ( ! is_array( $it ) ) {
				continue;
			}
			$title = isset( $it['title'] ) ? (string) $it['title'] : '';
			$link  = isset( $it['link'] ) ? (string) $it['link'] : '';
			if ( $title === '' || $link === '' ) {
				continue;
			}
			$items[] = array(
				'title' => $title,
				'link'  => $link,
			);
		}

		if ( $ttl > 0 ) {
			set_transient( $cache_key, $items, $ttl );
		}

		return $items;
	}
}
