<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class WP_AIW_Writer {
	private static $instance = null;

	/** @var WP_AIW_LLM_Client */
	private $llm;

	/** @var WP_AIW_Url_Fetcher */
	private $fetcher;

	public static function instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	private function __construct() {
		$this->llm = new WP_AIW_LLM_Client();
		$this->fetcher = new WP_AIW_Url_Fetcher();
	}

	private function get_output_length_hint( $language, $max_tokens ) {
		$language = ( $language === 'en' ) ? 'en' : 'zh';
		$max_tokens = (int) $max_tokens;
		if ( $max_tokens > 0 ) {
			return ( $language === 'en' )
				? ("\n\n[Output length constraint]\nThe output budget is approximately: {$max_tokens} tokens (hard limit; cannot be exceeded). If the content might be too long, proactively compress content_html (fewer paragraphs, shorter code samples, merge lists/tables, remove non-essential explanations), but ALWAYS return one complete, parseable, fully-closed JSON object. Never output partial JSON.")
				: ("\n\n【输出长度约束】\n本次输出最大长度上限约为：{$max_tokens}（硬上限，无法超过）。如可能超长，请主动压缩 content_html（减少段落、缩短示例代码、合并列表/表格、去掉非关键解释），但必须输出 1 个完整、可解析且闭合的 JSON；禁止输出半截 JSON。" );
		}

		return ( $language === 'en' )
			? "\n\n[Output length constraint]\nThis request does not specify max_tokens (provider/model default). Control your output length so you can reliably finish; prefer brevity over being cut off. ALWAYS return one complete, parseable, fully-closed JSON object. Never output partial JSON."
			: "\n\n【输出长度约束】\n本次未指定 max_tokens（使用模型默认）。请自行控制输出长度，宁可精简也不要因过长导致截断。必须输出 1 个完整、可解析且闭合的 JSON；禁止输出半截 JSON。";
	}

	/**
	 * Generate article content without writing to DB.
	 *
	 * @param array{title:string,requirements:string,language:string,urls:array<int,string>,reference_text:string} $input
	 * @return array{ok:bool, title?:string, content_html?:string, tags?:array<int,string>, debug?:array, error?:string}
	 */
	public function generate_content( $input ) {
		$settings = WP_AIW_Settings::instance()->get();

		$title = isset( $input['title'] ) ? trim( (string) $input['title'] ) : '';
		if ( $title === '' ) {
			return array( 'ok' => false, 'error' => '标题不能为空' );
		}

		$language = isset( $input['language'] ) ? strtolower( trim( (string) $input['language'] ) ) : 'zh';
		if ( ! in_array( $language, array( 'zh', 'en' ), true ) ) {
			$language = 'zh';
		}

		$requirements = isset( $input['requirements'] ) ? trim( (string) $input['requirements'] ) : '';
		$reference_text = isset( $input['reference_text'] ) ? trim( (string) $input['reference_text'] ) : '';

		$urls = array();
		if ( isset( $input['urls'] ) && is_array( $input['urls'] ) ) {
			foreach ( $input['urls'] as $u ) {
				$u = trim( (string) $u );
				if ( $u !== '' ) {
					$urls[] = $u;
				}
			}
			$urls = array_values( array_unique( $urls ) );
		}

		$max_urls = isset( $settings['url_fetch_max_urls'] ) ? (int) $settings['url_fetch_max_urls'] : 3;
		$max_urls = max( 1, min( 10, $max_urls ) );
		if ( count( $urls ) > $max_urls ) {
			$urls = array_slice( $urls, 0, $max_urls );
		}

		$url_timeout = isset( $settings['url_fetch_timeout_seconds'] ) ? (int) $settings['url_fetch_timeout_seconds'] : 15;
		$url_max_chars = isset( $settings['url_fetch_max_chars'] ) ? (int) $settings['url_fetch_max_chars'] : 12000;

		$references = array();
		$url_errors = array();
		foreach ( $urls as $u ) {
			$r = $this->fetcher->fetch( $u, $url_timeout, $url_max_chars );
			if ( empty( $r['ok'] ) ) {
				$url_errors[] = $u . '：' . ( isset( $r['error'] ) ? (string) $r['error'] : '抓取失败' );
				continue;
			}
			$references[] = $r['data'];
		}

		$references_block = '';
		if ( $reference_text !== '' ) {
			$references_block .= "\n\n[用户粘贴参考]\n" . $reference_text;
		}
		if ( ! empty( $references ) ) {
			foreach ( $references as $idx => $ref ) {
				$ref_title = isset( $ref['title'] ) ? trim( (string) $ref['title'] ) : '';
				$ref_text = isset( $ref['text'] ) ? trim( (string) $ref['text'] ) : '';
				$ref_url = isset( $ref['url'] ) ? trim( (string) $ref['url'] ) : '';
				$references_block .= "\n\n[URL参考 " . ( $idx + 1 ) . "]\nURL: " . $ref_url;
				if ( $ref_title !== '' ) {
					$references_block .= "\nTitle: " . $ref_title;
				}
				$references_block .= "\nContent: " . $ref_text;
			}
		}

		$summary_first = ! empty( $settings['writer_summary_first'] );
		$references_summary = '';
		$summary_debug = null;
		if ( $summary_first && $references_block !== '' ) {
			$summary = $this->summarize_references( $references_block, $language, $settings );
			if ( empty( $summary['ok'] ) ) {
				return array(
					'ok' => false,
					'error' => isset( $summary['error'] ) ? (string) $summary['error'] : '参考资料摘要失败',
					'debug' => array( 'url_errors' => $url_errors ),
				);
			}
			$references_summary = isset( $summary['summary_text'] ) ? (string) $summary['summary_text'] : '';
			$summary_debug = $summary;
		}

		$system_prompt = ( $language === 'en' ) ? (string) $settings['writer_prompt_system_en'] : (string) $settings['writer_prompt_system_zh'];
		$system_prompt = trim( $system_prompt );
		if ( $system_prompt === '' ) {
			$system_prompt = ( $language === 'en' ) ? 'You are an English technical writer. Output strict JSON with title/content_html/tags.' : '你是中文技术写作编辑。输出严格 JSON，包含 title/content_html/tags。';
		}

		$rules = ( $language === 'en' )
			? "\n\nOutput rules: output ONLY one JSON object. Do NOT wrap in Markdown code fences. Do NOT add any explanation text. Use valid JSON (double quotes, no trailing commas). Do NOT put literal newlines inside JSON strings; if needed, use \\n. Must include: title (string), content_html (string)."
			: "\n\n输出规则：只能输出 1 个 JSON 对象，不要使用 Markdown 代码块（不要输出 ```），不要输出任何解释文字。必须是严格 JSON（双引号、无尾逗号）。JSON 字符串里不要出现真实换行；如需换行请用 \\n。必须包含：title（string）与 content_html（string）。";

		$user = '';
		if ( $language === 'en' ) {
			$user .= "Title:\n" . $title . "\n\nRequirements:\n" . ( $requirements !== '' ? $requirements : '(none)' );
			if ( $references_summary !== '' ) {
				$user .= "\n\nReferences (summarized):\n" . $references_summary;
			} elseif ( $references_block !== '' ) {
				$user .= "\n\nReferences:\n" . $references_block;
			}
			$user .= $rules;
		} else {
			$user .= "标题：\n" . $title . "\n\n写作要求：\n" . ( $requirements !== '' ? $requirements : '（无）' );
			if ( $references_summary !== '' ) {
				$user .= "\n\n参考资料（已摘要）：\n" . $references_summary;
			} elseif ( $references_block !== '' ) {
				$user .= "\n\n参考资料：\n" . $references_block;
			}
			$user .= $rules;
		}

		$user .= $this->get_output_length_hint(
			$language,
			isset( $settings['llm_max_tokens'] ) ? (int) $settings['llm_max_tokens'] : 0
		);

		$messages = array(
			array( 'role' => 'system', 'content' => $system_prompt ),
			array( 'role' => 'user', 'content' => $user ),
		);

		$resp = $this->llm->chat_completions(
			$settings['llm_base_url'],
			$settings['llm_api_key'],
			$settings['llm_model'],
			$messages,
			isset( $settings['llm_timeout_seconds'] ) ? (int) $settings['llm_timeout_seconds'] : 60,
			isset( $settings['llm_max_tokens'] ) ? (int) $settings['llm_max_tokens'] : 0
		);

		if ( empty( $resp['ok'] ) ) {
			return array( 'ok' => false, 'error' => isset( $resp['error'] ) ? (string) $resp['error'] : 'LLM 调用失败' );
		}

		$text = '';
		$data = $resp['data'];
		if ( isset( $data['choices'][0]['message']['content'] ) ) {
			$text = (string) $data['choices'][0]['message']['content'];
		}
		$text = trim( $text );
		if ( $text === '' ) {
			return array( 'ok' => false, 'error' => 'LLM 返回为空' );
		}

		$parsed = $this->parse_llm_json( $text );
		if ( empty( $parsed['ok'] ) ) {
			return array( 'ok' => false, 'error' => isset( $parsed['error'] ) ? (string) $parsed['error'] : 'LLM 返回不是严格 JSON', 'debug' => array( 'raw' => $text ) );
		}

		$out_title = isset( $parsed['data']['title'] ) ? trim( (string) $parsed['data']['title'] ) : '';
		$out_html = isset( $parsed['data']['content_html'] ) ? trim( (string) $parsed['data']['content_html'] ) : '';
		if ( $out_title === '' ) {
			$out_title = $title;
		}
		if ( $out_html === '' ) {
			return array( 'ok' => false, 'error' => 'JSON 缺少 content_html', 'debug' => array( 'raw' => $text ) );
		}

		$tags = array();
		if ( isset( $parsed['data']['tags'] ) && is_array( $parsed['data']['tags'] ) ) {
			foreach ( $parsed['data']['tags'] as $t ) {
				$t = trim( (string) $t );
				if ( $t !== '' ) {
					$tags[] = $t;
				}
			}
			$tags = array_values( array_unique( $tags ) );
			if ( count( $tags ) > 20 ) {
				$tags = array_slice( $tags, 0, 20 );
			}
		}

		return array(
			'ok' => true,
			'title' => $out_title,
			'content_html' => $out_html,
			'tags' => $tags,
			'debug' => array(
				'urls_requested' => $urls,
				'url_errors' => $url_errors,
				'urls_fetched' => array_map( function ( $r ) { return isset( $r['url'] ) ? (string) $r['url'] : ''; }, $references ),
				'summary' => $summary_debug,
			),
		);
	}

	/**
	 * @param array{title:string,requirements:string,language:string,urls:array<int,string>,reference_text:string} $input
	 * @return array{ok:bool, post_id?:int, edit_url?:string, title?:string, tags?:array<int,string>, debug?:array, error?:string}
	 */
	public function generate_draft( $input ) {
		$settings = WP_AIW_Settings::instance()->get();

		$title = isset( $input['title'] ) ? trim( (string) $input['title'] ) : '';
		if ( $title === '' ) {
			return array( 'ok' => false, 'error' => '标题不能为空' );
		}

		$language = isset( $input['language'] ) ? strtolower( trim( (string) $input['language'] ) ) : 'zh';
		if ( ! in_array( $language, array( 'zh', 'en' ), true ) ) {
			$language = 'zh';
		}

		$requirements = isset( $input['requirements'] ) ? trim( (string) $input['requirements'] ) : '';
		$reference_text = isset( $input['reference_text'] ) ? trim( (string) $input['reference_text'] ) : '';

		$urls = array();
		if ( isset( $input['urls'] ) && is_array( $input['urls'] ) ) {
			foreach ( $input['urls'] as $u ) {
				$u = trim( (string) $u );
				if ( $u !== '' ) {
					$urls[] = $u;
				}
			}
			$urls = array_values( array_unique( $urls ) );
		}

		$max_urls = isset( $settings['url_fetch_max_urls'] ) ? (int) $settings['url_fetch_max_urls'] : 3;
		$max_urls = max( 1, min( 10, $max_urls ) );
		if ( count( $urls ) > $max_urls ) {
			$urls = array_slice( $urls, 0, $max_urls );
		}

		$url_timeout = isset( $settings['url_fetch_timeout_seconds'] ) ? (int) $settings['url_fetch_timeout_seconds'] : 15;
		$url_max_chars = isset( $settings['url_fetch_max_chars'] ) ? (int) $settings['url_fetch_max_chars'] : 12000;

		$references = array();
		$url_errors = array();
		foreach ( $urls as $u ) {
			$r = $this->fetcher->fetch( $u, $url_timeout, $url_max_chars );
			if ( empty( $r['ok'] ) ) {
				$url_errors[] = $u . '：' . ( isset( $r['error'] ) ? (string) $r['error'] : '抓取失败' );
				continue;
			}
			$references[] = $r['data'];
		}

		$references_block = '';
		if ( $reference_text !== '' ) {
			$references_block .= "\n\n[用户粘贴参考]\n" . $reference_text;
		}
		if ( ! empty( $references ) ) {
			foreach ( $references as $idx => $ref ) {
				$ref_title = isset( $ref['title'] ) ? trim( (string) $ref['title'] ) : '';
				$ref_text = isset( $ref['text'] ) ? trim( (string) $ref['text'] ) : '';
				$ref_url = isset( $ref['url'] ) ? trim( (string) $ref['url'] ) : '';
				$references_block .= "\n\n[URL参考 " . ( $idx + 1 ) . "]\nURL: " . $ref_url;
				if ( $ref_title !== '' ) {
					$references_block .= "\nTitle: " . $ref_title;
				}
				$references_block .= "\nContent: " . $ref_text;
			}
		}

		$summary_first = ! empty( $settings['writer_summary_first'] );
		$references_summary = '';
		$summary_debug = null;
		if ( $summary_first && $references_block !== '' ) {
			$summary = $this->summarize_references( $references_block, $language, $settings );
			if ( empty( $summary['ok'] ) ) {
				return array(
					'ok' => false,
					'error' => isset( $summary['error'] ) ? (string) $summary['error'] : '参考资料摘要失败',
					'debug' => array( 'url_errors' => $url_errors ),
				);
			}
			$references_summary = isset( $summary['summary_text'] ) ? (string) $summary['summary_text'] : '';
			$summary_debug = $summary;
		}

		$system_prompt = ( $language === 'en' ) ? (string) $settings['writer_prompt_system_en'] : (string) $settings['writer_prompt_system_zh'];
		$system_prompt = trim( $system_prompt );
		if ( $system_prompt === '' ) {
			$system_prompt = ( $language === 'en' ) ? 'You are an English technical writer. Output strict JSON with title/content_html/tags.' : '你是中文技术写作编辑。输出严格 JSON，包含 title/content_html/tags。';
		}

		$rules = ( $language === 'en' )
			? "\n\nOutput rules: output ONLY one JSON object. Do NOT wrap in Markdown code fences. Do NOT add any explanation text. Use valid JSON (double quotes, no trailing commas). Do NOT put literal newlines inside JSON strings; if needed, use \\n. Must include: title (string), content_html (string)."
			: "\n\n输出规则：只能输出 1 个 JSON 对象，不要使用 Markdown 代码块（不要输出 ```），不要输出任何解释文字。必须是严格 JSON（双引号、无尾逗号）。JSON 字符串里不要出现真实换行；如需换行请用 \\n。必须包含：title（string）与 content_html（string）。";

		$user = '';
		if ( $language === 'en' ) {
			$user .= "Title:\n" . $title . "\n\nRequirements:\n" . ( $requirements !== '' ? $requirements : '(none)' );
			if ( $references_summary !== '' ) {
				$user .= "\n\nReferences (summarized):\n" . $references_summary;
			} elseif ( $references_block !== '' ) {
				$user .= "\n\nReferences:\n" . $references_block;
			}
			$user .= $rules;
		} else {
			$user .= "标题：\n" . $title . "\n\n写作要求：\n" . ( $requirements !== '' ? $requirements : '（无）' );
			if ( $references_summary !== '' ) {
				$user .= "\n\n参考资料（已摘要）：\n" . $references_summary;
			} elseif ( $references_block !== '' ) {
				$user .= "\n\n参考资料：\n" . $references_block;
			}
			$user .= $rules;
		}

		$user .= $this->get_output_length_hint(
			$language,
			isset( $settings['llm_max_tokens'] ) ? (int) $settings['llm_max_tokens'] : 0
		);

		$messages = array(
			array( 'role' => 'system', 'content' => $system_prompt ),
			array( 'role' => 'user', 'content' => $user ),
		);

		$resp = $this->llm->chat_completions(
			$settings['llm_base_url'],
			$settings['llm_api_key'],
			$settings['llm_model'],
			$messages,
			isset( $settings['llm_timeout_seconds'] ) ? (int) $settings['llm_timeout_seconds'] : 60,
			isset( $settings['llm_max_tokens'] ) ? (int) $settings['llm_max_tokens'] : 0
		);

		if ( empty( $resp['ok'] ) ) {
			return array( 'ok' => false, 'error' => isset( $resp['error'] ) ? (string) $resp['error'] : 'LLM 调用失败' );
		}

		$text = '';
		$data = $resp['data'];
		if ( isset( $data['choices'][0]['message']['content'] ) ) {
			$text = (string) $data['choices'][0]['message']['content'];
		}
		$text = trim( $text );
		if ( $text === '' ) {
			return array( 'ok' => false, 'error' => 'LLM 返回为空' );
		}

		$parsed = $this->parse_llm_json( $text );
		if ( empty( $parsed['ok'] ) ) {
			return array( 'ok' => false, 'error' => isset( $parsed['error'] ) ? (string) $parsed['error'] : 'LLM 返回不是严格 JSON', 'debug' => array( 'raw' => $text ) );
		}

		$out_title = isset( $parsed['data']['title'] ) ? trim( (string) $parsed['data']['title'] ) : '';
		$out_html = isset( $parsed['data']['content_html'] ) ? trim( (string) $parsed['data']['content_html'] ) : '';
		if ( $out_title === '' ) {
			$out_title = $title;
		}
		if ( $out_html === '' ) {
			return array( 'ok' => false, 'error' => 'JSON 缺少 content_html', 'debug' => array( 'raw' => $text ) );
		}

		$tags = array();
		if ( isset( $parsed['data']['tags'] ) && is_array( $parsed['data']['tags'] ) ) {
			foreach ( $parsed['data']['tags'] as $t ) {
				$t = trim( (string) $t );
				if ( $t !== '' ) {
					$tags[] = $t;
				}
			}
			$tags = array_values( array_unique( $tags ) );
			if ( count( $tags ) > 20 ) {
				$tags = array_slice( $tags, 0, 20 );
			}
		}

		$postarr = array(
			'post_title' => $out_title,
			'post_content' => $out_html,
			'post_status' => 'draft',
			'post_type' => 'post',
			'post_author' => get_current_user_id(),
		);
		$post_id = wp_insert_post( $postarr, true );
		if ( is_wp_error( $post_id ) ) {
			return array( 'ok' => false, 'error' => $post_id->get_error_message() );
		}

		if ( ! empty( $tags ) ) {
			wp_set_post_tags( $post_id, $tags, false );
		}

		$edit_url = get_edit_post_link( $post_id, 'raw' );
		if ( ! is_string( $edit_url ) ) {
			$edit_url = '';
		}

		return array(
			'ok' => true,
			'post_id' => (int) $post_id,
			'edit_url' => $edit_url,
			'title' => $out_title,
			'tags' => $tags,
			'debug' => array(
				'urls_requested' => $urls,
				'url_errors' => $url_errors,
				'urls_fetched' => array_map( function ( $r ) { return isset( $r['url'] ) ? (string) $r['url'] : ''; }, $references ),
				'summary' => $summary_debug,
			),
		);
	}

	/**
	 * Generate draft from URLs only (title is generated by the model).
	 *
	 * @param array{requirements:string,reference_text?:string,language:string,urls:array<int,string>} $input
	 * @return array{ok:bool, post_id?:int, edit_url?:string, title?:string, tags?:array<int,string>, debug?:array, error?:string}
	 */
	public function generate_draft_from_urls( $input ) {
		$settings = WP_AIW_Settings::instance()->get();

		$language = isset( $input['language'] ) ? strtolower( trim( (string) $input['language'] ) ) : 'zh';
		if ( ! in_array( $language, array( 'zh', 'en' ), true ) ) {
			$language = 'zh';
		}

		$requirements = isset( $input['requirements'] ) ? trim( (string) $input['requirements'] ) : '';
		$reference_text = isset( $input['reference_text'] ) ? trim( (string) $input['reference_text'] ) : '';

		$urls = array();
		if ( isset( $input['urls'] ) && is_array( $input['urls'] ) ) {
			foreach ( $input['urls'] as $u ) {
				$u = trim( (string) $u );
				if ( $u !== '' ) {
					$urls[] = $u;
				}
			}
			$urls = array_values( array_unique( $urls ) );
		}

		// Product requirement: max 3 URLs.
		if ( count( $urls ) > 3 ) {
			$urls = array_slice( $urls, 0, 3 );
		}
		if ( empty( $urls ) ) {
			return array( 'ok' => false, 'error' => 'URL 不能为空（最多 3 个）' );
		}

		$url_timeout = isset( $settings['url_fetch_timeout_seconds'] ) ? (int) $settings['url_fetch_timeout_seconds'] : 15;
		$url_max_chars = isset( $settings['url_fetch_max_chars'] ) ? (int) $settings['url_fetch_max_chars'] : 12000;

		$references = array();
		$url_errors = array();
		foreach ( $urls as $u ) {
			$r = $this->fetcher->fetch( $u, $url_timeout, $url_max_chars );
			if ( empty( $r['ok'] ) ) {
				$url_errors[] = $u . '：' . ( isset( $r['error'] ) ? (string) $r['error'] : '抓取失败' );
				continue;
			}
			$references[] = $r['data'];
		}

		if ( empty( $references ) ) {
			return array( 'ok' => false, 'error' => 'URL 抓取失败（无可用参考内容）', 'debug' => array( 'urls_requested' => $urls, 'url_errors' => $url_errors ) );
		}

		$references_block = '';
		if ( $reference_text !== '' ) {
			$references_block .= "\n\n[用户粘贴参考]\n" . $reference_text;
		}
		foreach ( $references as $idx => $ref ) {
			$ref_title = isset( $ref['title'] ) ? trim( (string) $ref['title'] ) : '';
			$ref_text = isset( $ref['text'] ) ? trim( (string) $ref['text'] ) : '';
			$ref_url = isset( $ref['url'] ) ? trim( (string) $ref['url'] ) : '';
			$references_block .= "\n\n[URL参考 " . ( $idx + 1 ) . "]\nURL: " . $ref_url;
			if ( $ref_title !== '' ) {
				$references_block .= "\nTitle: " . $ref_title;
			}
			$references_block .= "\nContent: " . $ref_text;
		}

		$system_prompt = ( $language === 'en' ) ? (string) $settings['writer_prompt_system_en'] : (string) $settings['writer_prompt_system_zh'];
		$system_prompt = trim( $system_prompt );
		if ( $system_prompt === '' ) {
			$system_prompt = ( $language === 'en' ) ? 'You are an English technical writer. Output strict JSON with title/content_html/tags.' : '你是中文技术写作编辑。输出严格 JSON，包含 title/content_html/tags。';
		}

		$rules = ( $language === 'en' )
			? "\n\nOutput rules: output ONLY one JSON object. Do NOT wrap in Markdown code fences. Do NOT add any explanation text. Use valid JSON (double quotes, no trailing commas). Do NOT put literal newlines inside JSON strings; if needed, use \\n. Must include: title (string), content_html (string)."
			: "\n\n输出规则：只能输出 1 个 JSON 对象，不要使用 Markdown 代码块（不要输出 ```），不要输出任何解释文字。必须是严格 JSON（双引号、无尾逗号）。JSON 字符串里不要出现真实换行；如需换行请用 \\n。必须包含：title（string）与 content_html（string）。";

		$user = '';
		if ( $language === 'en' ) {
			$user .= "Task:\nBased on the references, generate a suitable title and a complete article.\n\nRequirements:\n" . ( $requirements !== '' ? $requirements : '(none)' );
			$user .= "\n\nReferences:\n" . $references_block;
			$user .= $rules;
		} else {
			$user .= "任务：\n根据参考资料，生成合适的标题与完整文章。\n\n写作要求：\n" . ( $requirements !== '' ? $requirements : '（无）' );
			$user .= "\n\n参考资料：\n" . $references_block;
			$user .= $rules;
		}

		$user .= $this->get_output_length_hint(
			$language,
			isset( $settings['llm_max_tokens'] ) ? (int) $settings['llm_max_tokens'] : 0
		);

		$messages = array(
			array( 'role' => 'system', 'content' => $system_prompt ),
			array( 'role' => 'user', 'content' => $user ),
		);

		$resp = $this->llm->chat_completions(
			$settings['llm_base_url'],
			$settings['llm_api_key'],
			$settings['llm_model'],
			$messages,
			isset( $settings['llm_timeout_seconds'] ) ? (int) $settings['llm_timeout_seconds'] : 60,
			isset( $settings['llm_max_tokens'] ) ? (int) $settings['llm_max_tokens'] : 0
		);

		if ( empty( $resp['ok'] ) ) {
			return array( 'ok' => false, 'error' => isset( $resp['error'] ) ? (string) $resp['error'] : 'LLM 调用失败' );
		}

		$text = '';
		$data = $resp['data'];
		if ( isset( $data['choices'][0]['message']['content'] ) ) {
			$text = (string) $data['choices'][0]['message']['content'];
		}
		$text = trim( $text );
		if ( $text === '' ) {
			return array( 'ok' => false, 'error' => 'LLM 返回为空' );
		}

		$parsed = $this->parse_llm_json( $text );
		if ( empty( $parsed['ok'] ) ) {
			return array( 'ok' => false, 'error' => isset( $parsed['error'] ) ? (string) $parsed['error'] : 'LLM 返回不是严格 JSON', 'debug' => array( 'raw' => $text ) );
		}

		$out_title = isset( $parsed['data']['title'] ) ? trim( (string) $parsed['data']['title'] ) : '';
		$out_html = isset( $parsed['data']['content_html'] ) ? trim( (string) $parsed['data']['content_html'] ) : '';
		if ( $out_title === '' ) {
			$out_title = ( $language === 'en' ) ? 'Untitled' : '未命名';
		}
		if ( $out_html === '' ) {
			return array( 'ok' => false, 'error' => 'JSON 缺少 content_html', 'debug' => array( 'raw' => $text ) );
		}

		$tags = array();
		if ( isset( $parsed['data']['tags'] ) && is_array( $parsed['data']['tags'] ) ) {
			foreach ( $parsed['data']['tags'] as $t ) {
				$t = trim( (string) $t );
				if ( $t !== '' ) {
					$tags[] = $t;
				}
			}
			$tags = array_values( array_unique( $tags ) );
			if ( count( $tags ) > 20 ) {
				$tags = array_slice( $tags, 0, 20 );
			}
		}

		$post_id = wp_insert_post(
			array(
				'post_title' => $out_title,
				'post_content' => $out_html,
				'post_status' => 'draft',
				'post_type' => 'post',
			),
			true
		);

		if ( is_wp_error( $post_id ) ) {
			return array( 'ok' => false, 'error' => $post_id->get_error_message() );
		}

		if ( ! empty( $tags ) ) {
			wp_set_post_tags( (int) $post_id, $tags, false );
		}

		return array(
			'ok' => true,
			'post_id' => (int) $post_id,
			'edit_url' => get_edit_post_link( (int) $post_id, 'raw' ),
			'title' => $out_title,
			'tags' => $tags,
			'debug' => array(
				'urls_requested' => $urls,
				'url_errors' => $url_errors,
				'urls_fetched' => array_map( function ( $r ) { return isset( $r['url'] ) ? (string) $r['url'] : ''; }, $references ),
			),
		);
	}

	/**
	 * @return array{ok:bool, summary_text?:string, error?:string, raw?:string}
	 */
	private function summarize_references( $references_block, $language, $settings ) {
		$language = ( $language === 'en' ) ? 'en' : 'zh';
		$references_block = is_string( $references_block ) ? trim( $references_block ) : '';
		if ( $references_block === '' ) {
			return array( 'ok' => true, 'summary_text' => '' );
		}

		$system = ( $language === 'en' )
			? 'You summarize technical references for writing. Output strict JSON only.'
			: '你是技术资料整理助手。请对参考资料做摘要与要点提取，只输出严格 JSON。';

		$rules = ( $language === 'en' )
			? "Output ONLY one JSON object. Fields: summary_text (string, required), key_points (array of strings, optional). No Markdown."
			: "只能输出 1 个 JSON 对象。字段：summary_text（string，必填）、key_points（string 数组，可选）。不要 Markdown。";

		$user = ( $language === 'en' )
			? "References:\n" . $references_block . "\n\n" . $rules
			: "参考资料：\n" . $references_block . "\n\n" . $rules;

		$messages = array(
			array( 'role' => 'system', 'content' => $system ),
			array( 'role' => 'user', 'content' => $user ),
		);

		$resp = $this->llm->chat_completions(
			$settings['llm_base_url'],
			$settings['llm_api_key'],
			$settings['llm_model'],
			$messages,
			isset( $settings['llm_timeout_seconds'] ) ? (int) $settings['llm_timeout_seconds'] : 60,
			isset( $settings['llm_max_tokens'] ) ? (int) $settings['llm_max_tokens'] : 0
		);

		if ( empty( $resp['ok'] ) ) {
			return array( 'ok' => false, 'error' => isset( $resp['error'] ) ? (string) $resp['error'] : 'LLM 调用失败' );
		}

		$text = '';
		$data = $resp['data'];
		if ( isset( $data['choices'][0]['message']['content'] ) ) {
			$text = (string) $data['choices'][0]['message']['content'];
		}
		$text = trim( $text );
		if ( $text === '' ) {
			return array( 'ok' => false, 'error' => 'LLM 返回为空' );
		}

		$parsed = $this->parse_llm_json( $text );
		if ( empty( $parsed['ok'] ) ) {
			return array( 'ok' => false, 'error' => isset( $parsed['error'] ) ? (string) $parsed['error'] : '摘要返回不是严格 JSON', 'raw' => $text );
		}

		$summary_text = isset( $parsed['data']['summary_text'] ) ? trim( (string) $parsed['data']['summary_text'] ) : '';
		if ( $summary_text === '' ) {
			return array( 'ok' => false, 'error' => '摘要 JSON 缺少 summary_text', 'raw' => $text );
		}

		return array(
			'ok' => true,
			'summary_text' => $summary_text,
			'raw' => $text,
		);
	}

	/**
	 * @return array{ok:bool, data?:array, error?:string}
	 */
	private function parse_llm_json( $text ) {
		$text = is_string( $text ) ? trim( $text ) : '';
		if ( $text === '' ) {
			return array( 'ok' => false, 'error' => 'LLM 返回为空' );
		}

		$json_text = $this->extract_first_json_object( $text );
		$parsed = $json_text !== '' ? json_decode( $json_text, true ) : null;
		if ( ! is_array( $parsed ) ) {
			$sanitized = $json_text !== '' ? $this->sanitize_json_string_literals( $json_text ) : '';
			if ( $sanitized !== '' ) {
				$parsed = json_decode( $sanitized, true );
			}
			if ( ! is_array( $parsed ) ) {
				$err = function_exists( 'json_last_error_msg' ) ? json_last_error_msg() : 'JSON parse error';
				return array( 'ok' => false, 'error' => 'LLM 返回不是严格 JSON：' . $err . '（请检查提示词或模型输出）' );
			}
		}

		return array( 'ok' => true, 'data' => $parsed );
	}

	/**
	 * Extract the first JSON object from LLM output.
	 * Supports ```json fenced blocks and surrounding text.
	 */
	private function extract_first_json_object( $text ) {
		$text = is_string( $text ) ? trim( $text ) : '';
		if ( $text === '' ) {
			return '';
		}

		if ( preg_match( '/```(?:json)?\s*([\s\S]*?)\s*```/i', $text, $m ) ) {
			$candidate = trim( (string) $m[1] );
			if ( $candidate !== '' ) {
				$text = $candidate;
			}
		}

		$start = strpos( $text, '{' );
		if ( $start === false ) {
			return '';
		}

		$depth = 0;
		$in_string = false;
		$escape = false;
		$len = strlen( $text );
		for ( $i = $start; $i < $len; $i++ ) {
			$ch = $text[ $i ];
			if ( $in_string ) {
				if ( $escape ) {
					$escape = false;
					continue;
				}
				if ( $ch === '\\' ) {
					$escape = true;
					continue;
				}
				if ( $ch === '"' ) {
					$in_string = false;
				}
				continue;
			}

			if ( $ch === '"' ) {
				$in_string = true;
				continue;
			}

			if ( $ch === '{' ) {
				$depth++;
				continue;
			}
			if ( $ch === '}' ) {
				$depth--;
				if ( $depth === 0 ) {
					return substr( $text, $start, $i - $start + 1 );
				}
			}
		}

		return '';
	}

	/**
	 * Sanitize JSON strings that contain literal newlines/tabs inside quoted strings.
	 */
	private function sanitize_json_string_literals( $json ) {
		$json = is_string( $json ) ? $json : '';
		if ( $json === '' ) {
			return '';
		}

		$out = '';
		$in_string = false;
		$escape = false;
		$len = strlen( $json );
		for ( $i = 0; $i < $len; $i++ ) {
			$ch = $json[ $i ];
			if ( $in_string ) {
				if ( $escape ) {
					$escape = false;
					$out .= $ch;
					continue;
				}
				if ( $ch === '\\' ) {
					$escape = true;
					$out .= $ch;
					continue;
				}
				if ( $ch === '"' ) {
					$in_string = false;
					$out .= $ch;
					continue;
				}
				if ( $ch === "\n" ) {
					$out .= "\\n";
					continue;
				}
				if ( $ch === "\r" ) {
					continue;
				}
				if ( $ch === "\t" ) {
					$out .= "\\t";
					continue;
				}
				$out .= $ch;
				continue;
			}

			if ( $ch === '"' ) {
				$in_string = true;
				$out .= $ch;
				continue;
			}
			$out .= $ch;
		}

		return $out;
	}
}
