<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class WP_AIW_Writer_Metabox {
	private static $instance = null;

	private function get_supported_post_types() {
		$post_types = get_post_types( array( 'show_ui' => true ), 'names' );
		if ( ! is_array( $post_types ) ) {
			$post_types = array();
		}

		$allowed = array();
		foreach ( $post_types as $pt ) {
			$pt = (string) $pt;
			if ( $pt === '' || $pt === 'attachment' ) {
				continue;
			}
			if ( ! post_type_supports( $pt, 'editor' ) ) {
				continue;
			}
			$allowed[] = $pt;
		}

		$allowed = array_values( array_unique( $allowed ) );
		/**
		 * Filter supported post types for the AI Writer metabox.
		 *
		 * @param string[] $allowed Post type slugs.
		 */
		return apply_filters( 'wp_aiw_writer_metabox_post_types', $allowed );
	}

	public static function instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	private function __construct() {
		add_action( 'add_meta_boxes', array( $this, 'register_metabox' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_assets' ) );
	}

	public function register_metabox() {
		if ( ! current_user_can( 'edit_posts' ) && ! current_user_can( 'edit_pages' ) ) {
			return;
		}

		$post_types = $this->get_supported_post_types();
		if ( empty( $post_types ) ) {
			return;
		}

		add_meta_box(
			'wp-aiw-writer-metabox',
			'AI写文章/重写文章',
			array( $this, 'render' ),
			$post_types,
			'side',
			'high'
		);
	}

	public function enqueue_assets( $hook ) {
		if ( ! in_array( $hook, array( 'post.php', 'post-new.php' ), true ) ) {
			return;
		}

		$screen = function_exists( 'get_current_screen' ) ? get_current_screen() : null;
		$post_types = $this->get_supported_post_types();
		if ( ! $screen || empty( $screen->post_type ) || empty( $post_types ) || ! in_array( $screen->post_type, $post_types, true ) ) {
			return;
		}

		if ( ! current_user_can( 'edit_posts' ) && ! current_user_can( 'edit_pages' ) ) {
			return;
		}

		$settings = WP_AIW_Settings::instance()->get();

		wp_enqueue_script( 'wp-aiw-writer-metabox', WP_AIW_URL . 'assets/js/writer-metabox.js', array(), WP_AIW_VERSION, true );
		wp_localize_script(
			'wp-aiw-writer-metabox',
			'wpAiwWriterMetabox',
			array(
				'ajaxUrl' => admin_url( 'admin-ajax.php' ),
				'nonce' => wp_create_nonce( 'wp_aiw_nonce' ),
				'image' => array(
					'provider' => isset( $settings['image_provider'] ) ? (string) $settings['image_provider'] : 'gemini_generatecontent',
					'enabled' => ! empty( $settings['image_enabled'] ) ? 1 : 0,
					'defaultSize' => isset( $settings['image_size'] ) ? (string) $settings['image_size'] : '1024x1024',
					'maxPerPost' => isset( $settings['image_max_per_post'] ) ? (int) $settings['image_max_per_post'] : 3,
				),
			)
		);
	}

	public function render( $post ) {
		if ( ! $post || ! isset( $post->ID ) ) {
			return;
		}

		$settings = WP_AIW_Settings::instance()->get();
		?>
		<div id="wp-aiw-writer-metabox">
			<input type="hidden" id="wp-aiw-writer-mb-post-id" value="<?php echo esc_attr( (string) $post->ID ); ?>" />
			<p style="margin-top:0;">支持两种模式：写入当前文章（正文可覆盖/追加，tags 合并写入，写入后需手动保存），或对当前文章进行重写优化（覆盖正文与 tags，写入后需手动保存）。</p>

			<p>
				<label for="wp-aiw-writer-mb-mode" style="font-weight:600; display:block;">模式</label>
				<select id="wp-aiw-writer-mb-mode" class="widefat">
					<option value="write_current">写入当前文章（不自动保存）</option>
					<option value="rewrite_current">重写当前文章（优化，不自动保存）</option>
				</select>
			</p>

			<div id="wp-aiw-writer-mb-write-options" style="display:none; margin: 6px 0 0 0; padding: 8px; background:#f6f7f7; border:1px solid #dcdcde; border-radius:4px;">
				<p style="margin:0 0 8px 0;">
					<label for="wp-aiw-writer-mb-write-mode" style="font-weight:600; display:block;">写入方式</label>
					<select id="wp-aiw-writer-mb-write-mode" class="widefat">
						<option value="overwrite">覆盖正文</option>
						<option value="append">追加到正文末尾</option>
					</select>
				</p>
				<p style="margin:0;">
					<label>
						<input type="checkbox" id="wp-aiw-writer-mb-write-title" value="1" /> 同时覆盖文章标题
					</label>
				</p>
			</div>

			<div id="wp-aiw-writer-mb-rewrite-options" style="display:none; margin: 6px 0 0 0; padding: 8px; background:#f6f7f7; border:1px solid #dcdcde; border-radius:4px;">
				<p style="margin:0;">
					<label>
						<input type="checkbox" id="wp-aiw-writer-mb-rewrite-title" value="1" /> 同时重写文章标题
					</label>
				</p>
			</div>

			<p>
				<label for="wp-aiw-writer-mb-title" style="font-weight:600; display:block;">标题</label>
				<input id="wp-aiw-writer-mb-title" type="text" class="widefat" placeholder="默认读取当前文章标题" />
			</p>

			<p>
				<label for="wp-aiw-writer-mb-req" style="font-weight:600; display:block;">写作要求</label>
				<textarea id="wp-aiw-writer-mb-req" class="widefat" rows="5" placeholder="例如：技术教程，步骤清晰，包含示例代码与常见坑"></textarea>
			</p>

			<p>
				<label for="wp-aiw-writer-mb-urls" style="font-weight:600; display:block;">参考 URL（可选）</label>
				<textarea id="wp-aiw-writer-mb-urls" class="widefat" rows="4" placeholder="每行一个 URL"></textarea>
				<span class="description">最多 <?php echo esc_html( (string) $settings['url_fetch_max_urls'] ); ?> 个</span>
			</p>

			<p>
				<label for="wp-aiw-writer-mb-ref" style="font-weight:600; display:block;">粘贴参考（可选）</label>
				<textarea id="wp-aiw-writer-mb-ref" class="widefat" rows="4" placeholder="可粘贴笔记/大纲/参考资料"></textarea>
			</p>

			<p>
				<label for="wp-aiw-writer-mb-lang" style="font-weight:600; display:block;">语言</label>
				<select id="wp-aiw-writer-mb-lang" class="widefat">
					<option value="zh">中文</option>
					<option value="en">English</option>
				</select>
			</p>

			<div id="wp-aiw-writer-mb-image" style="margin:10px 0 0 0; padding:8px; background:#fff; border:1px solid #dcdcde; border-radius:4px;">
				<p style="margin:0 0 6px 0;"><strong>插图（文生图）</strong></p>
				<p style="margin:0 0 8px 0; color:#50575e;">默认关闭。启用后可生成预览，并可选入库后插入正文。</p>
				<p style="margin:0 0 8px 0;">
					<label for="wp-aiw-writer-mb-image-style" style="font-weight:600; display:block;">风格（可选）</label>
					<select id="wp-aiw-writer-mb-image-style" class="widefat">
						<option value="">默认（不额外指定）</option>
						<option value="modern flat tech illustration, clean, no text, no watermark">现代扁平科技插画（无文字）</option>
						<option value="simple line art diagram, monochrome, clean, no text">黑白线框图（diagram）</option>
						<option value="isometric illustration, clean, modern, no text">等距/isometric 插画（无文字）</option>
						<option value="minimal UI-style illustration, clean, no text">极简 UI 风格（无文字）</option>
					</select>
				</p>
				<p style="margin:0 0 8px 0;">
					<label for="wp-aiw-writer-mb-image-excerpt" style="font-weight:600; display:block;">内容摘抄提示词（可选）</label>
					<textarea id="wp-aiw-writer-mb-image-excerpt" class="widefat" rows="4" placeholder="可从当前文章中摘抄关键段落/步骤/要点，用于让插图更贴合内容（建议 200~1500 字符）"></textarea>
				</p>
				<p style="margin:0 0 8px 0;">
					<label for="wp-aiw-writer-mb-image-size" style="font-weight:600; display:block;">尺寸</label>
					<select id="wp-aiw-writer-mb-image-size" class="widefat">
						<option value="">默认（设置页）</option>
						<option value="2K">2K（如火山 Ark）</option>
						<option value="1024x1024">1024x1024</option>
						<option value="1024x1536">1024x1536</option>
						<option value="1536x1024">1536x1024</option>
						<option value="512x512">512x512</option>
					</select>
				</p>
				<p style="margin:0 0 8px 0;">
					<button type="button" class="button" id="wp-aiw-writer-mb-image-generate">生成插图预览</button>
					<button type="button" class="button" id="wp-aiw-writer-mb-image-insert" disabled>插入正文</button>
					<button type="button" class="button" id="wp-aiw-writer-mb-image-featured" disabled>设为特色图</button>
					<span id="wp-aiw-writer-mb-image-status" style="margin-left:8px;"></span>
				</p>
				<div id="wp-aiw-writer-mb-image-preview" style="display:none; margin:0;">
					<div style="margin:0 0 6px 0; color:#50575e;">预览：</div>
					<img id="wp-aiw-writer-mb-image-preview-img" src="" alt="" style="max-width:100%; height:auto; display:block; border:1px solid #dcdcde; border-radius:4px;" />
				</div>
			</div>

			<p>
				<label>
					<input type="checkbox" id="wp-aiw-writer-mb-debug" value="1" /> 返回调试信息
				</label>
			</p>

			<p>
				<button type="button" class="button button-primary" id="wp-aiw-writer-mb-generate">生成并写入</button>
				<span id="wp-aiw-writer-mb-status" style="margin-left:8px;"></span>
			</p>

			<div id="wp-aiw-writer-mb-result" style="margin-top:8px;"></div>
			<div id="wp-aiw-writer-mb-log" style="margin-top:8px; padding:8px; border:1px solid #dcdcde; border-radius:4px; background:#fff; white-space: pre-wrap;"></div>
		</div>
		<?php
	}
}
