<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class WP_AIW_Writer_Admin {
	private static $instance = null;

	public static function instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	private function __construct() {
		add_action( 'admin_menu', array( $this, 'register_pages' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_assets' ) );
		add_action( 'wp_ajax_wp_aiw_writer_generate_draft', array( $this, 'ajax_generate_draft' ) );
		add_action( 'wp_ajax_wp_aiw_writer_generate_draft_from_urls', array( $this, 'ajax_generate_draft_from_urls' ) );
		add_action( 'wp_ajax_wp_aiw_writer_generate_content', array( $this, 'ajax_generate_content' ) );
		add_action( 'wp_ajax_wp_aiw_rewrite_generate_current', array( $this, 'ajax_generate_rewrite_current' ) );
	}

	public function register_pages() {
		add_submenu_page(
			'wp-aiw',
			'AI 写文章',
			'AI 写文章',
			'manage_options',
			'wp-aiw-writer',
			array( $this, 'render_writer_page' )
		);

		add_submenu_page(
			'wp-aiw',
			'URL 抓取文章',
			'URL 抓取文章',
			'manage_options',
			'wp-aiw-writer-from-url',
			array( $this, 'render_writer_from_url_page' )
		);

		add_submenu_page(
			'wp-aiw',
			'AI 批量写文章',
			'AI 批量写文章',
			'manage_options',
			'wp-aiw-writer-batch',
			array( $this, 'render_writer_batch_page' )
		);
	}

	public function enqueue_assets( $hook ) {
		$page = isset( $_GET['page'] ) ? sanitize_key( wp_unslash( $_GET['page'] ) ) : '';
		if ( $page !== 'wp-aiw-writer' && $page !== 'wp-aiw-writer-from-url' && $page !== 'wp-aiw-writer-batch' ) {
			return;
		}

		if ( $page === 'wp-aiw-writer-batch' ) {
			$settings = WP_AIW_Settings::instance()->get();
			wp_enqueue_script( 'wp-aiw-featured-image', WP_AIW_URL . 'assets/js/featured-image.js', array(), WP_AIW_VERSION, true );
			wp_enqueue_script( 'wp-aiw-writer-batch', WP_AIW_URL . 'assets/js/writer-batch.js', array( 'wp-aiw-featured-image' ), WP_AIW_VERSION, true );
			wp_localize_script(
				'wp-aiw-writer-batch',
				'wpAiwWriterBatch',
				array(
					'ajaxUrl' => admin_url( 'admin-ajax.php' ),
					'nonce' => wp_create_nonce( 'wp_aiw_nonce' ),
					'defaultConcurrency' => isset( $settings['writer_batch_size'] ) ? (int) $settings['writer_batch_size'] : 1,
					'imageEnabled' => ! empty( $settings['image_enabled'] ) ? 1 : 0,
				)
			);
			return;
		}

		$settings = WP_AIW_Settings::instance()->get();
		wp_enqueue_script( 'wp-aiw-featured-image', WP_AIW_URL . 'assets/js/featured-image.js', array(), WP_AIW_VERSION, true );

		$handle = 'wp-aiw-writer';
		$src = WP_AIW_URL . 'assets/js/writer.js';
		if ( $page === 'wp-aiw-writer-from-url' ) {
			$handle = 'wp-aiw-writer-from-url';
			$src = WP_AIW_URL . 'assets/js/writer-from-url.js';
		}

		wp_enqueue_script( $handle, $src, array( 'wp-aiw-featured-image' ), WP_AIW_VERSION, true );
		wp_localize_script(
			$handle,
			'wpAiwWriter',
			array(
				'ajaxUrl' => admin_url( 'admin-ajax.php' ),
				'nonce' => wp_create_nonce( 'wp_aiw_nonce' ),
				'imageEnabled' => ! empty( $settings['image_enabled'] ) ? 1 : 0,
			)
		);
	}

	public function render_writer_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$settings = WP_AIW_Settings::instance()->get();
		$default_lang = 'zh';
		?>
		<div class="wrap">
			<h1>AI Writer / AI 写文章</h1>
			<p>输入标题与写作要求，可选填参考 URL（插件会抓取并清洗）。生成后会创建一篇 <strong>草稿</strong> 并返回编辑链接。</p>

			<table class="form-table" role="presentation">
				<tr>
					<th scope="row"><label for="wp-aiw-writer-title">标题</label></th>
					<td>
						<input id="wp-aiw-writer-title" type="text" class="regular-text" placeholder="例如：WordPress 插件如何安全调用 OpenAI-compatible 接口" />
					</td>
				</tr>
				<tr>
					<th scope="row"><label for="wp-aiw-writer-req">写作要求</label></th>
					<td>
						<textarea id="wp-aiw-writer-req" class="large-text" rows="6" placeholder="例如：面向初学者，包含步骤、示例代码、常见坑与排查方式。"></textarea>
						<p class="description">偏技术教程风格（A），结构清晰、可执行步骤。</p>
					</td>
				</tr>
				<tr>
					<th scope="row"><label for="wp-aiw-writer-urls">参考 URL（可选）</label></th>
					<td>
						<textarea id="wp-aiw-writer-urls" class="large-text" rows="4" placeholder="每行一个 URL"></textarea>
						<p class="description">最多抓取 <?php echo esc_html( (string) $settings['url_fetch_max_urls'] ); ?> 个；单个 URL 最大提取字符 <?php echo esc_html( (string) $settings['url_fetch_max_chars'] ); ?>。</p>
					</td>
				</tr>
				<tr>
					<th scope="row"><label for="wp-aiw-writer-ref">粘贴参考内容（可选）</label></th>
					<td>
						<textarea id="wp-aiw-writer-ref" class="large-text" rows="5" placeholder="可直接粘贴参考资料/大纲/笔记"></textarea>
					</td>
				</tr>
				<tr>
					<th scope="row"><label for="wp-aiw-writer-lang">语言</label></th>
					<td>
						<select id="wp-aiw-writer-lang">
							<option value="zh" <?php selected( $default_lang, 'zh' ); ?>>中文</option>
							<option value="en" <?php selected( $default_lang, 'en' ); ?>>English</option>
						</select>
						<label style="margin-left: 12px;">
							<input type="checkbox" id="wp-aiw-writer-debug" value="1" /> 返回调试信息
						</label>
					</td>
				</tr>
				<tr>
					<th scope="row">自动特色图（可选）</th>
					<td>
						<label>
							<input type="checkbox" id="wp-aiw-writer-auto-featured" value="1" />
							生成预览图 → 下载入库 → 设为特色图（失败不影响草稿）
						</label>
						<p class="description">图片提示词仅使用标题。需在设置页开启 image_enabled。</p>
					</td>
				</tr>
			</table>

			<p>
				<button id="wp-aiw-writer-generate" class="button button-primary">生成草稿</button>
				<span id="wp-aiw-writer-status" style="margin-left: 10px;"></span>
			</p>

			<div id="wp-aiw-writer-result" style="margin-top: 12px;"></div>
			<div id="wp-aiw-writer-log" style="margin-top: 12px; padding: 12px; background: #fff; border: 1px solid #dcdcde; border-radius: 6px; white-space: pre-wrap;"></div>
		</div>
		<?php
	}

	public function render_writer_from_url_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$default_lang = 'zh';
		?>
		<div class="wrap">
			<h1>URL 抓取文章</h1>
			<p>仅输入 URL（最多 3 个），插件会抓取并清洗内容，然后由模型生成 <strong>标题 + 正文 + tags</strong> 并创建一篇 <strong>草稿</strong>。</p>

			<table class="form-table" role="presentation">
				<tr>
					<th scope="row"><label for="wp-aiw-writer-url-urls">URL（必填）</label></th>
					<td>
						<textarea id="wp-aiw-writer-url-urls" class="large-text" rows="5" placeholder="每行一个 URL，最多 3 个"></textarea>
						<p class="description">多 URL 会合并生成 1 篇文章。</p>
					</td>
				</tr>
				<tr>
					<th scope="row"><label for="wp-aiw-writer-url-ref">粘贴参考内容（可选）</label></th>
					<td>
						<textarea id="wp-aiw-writer-url-ref" class="large-text" rows="6" placeholder="可粘贴要合并进文章的参考内容/笔记/大纲"></textarea>
					</td>
				</tr>
				<tr>
					<th scope="row"><label for="wp-aiw-writer-url-req">写作要求（可选）</label></th>
					<td>
						<textarea id="wp-aiw-writer-url-req" class="large-text" rows="5" placeholder="例如：总结要点 + 给出行动建议 + 适当引用数据。"></textarea>
					</td>
				</tr>
				<tr>
					<th scope="row"><label for="wp-aiw-writer-url-lang">语言</label></th>
					<td>
						<select id="wp-aiw-writer-url-lang">
							<option value="zh" <?php selected( $default_lang, 'zh' ); ?>>中文</option>
							<option value="en" <?php selected( $default_lang, 'en' ); ?>>English</option>
						</select>
						<label style="margin-left: 12px;">
							<input type="checkbox" id="wp-aiw-writer-url-debug" value="1" /> 返回调试信息
						</label>
					</td>
				</tr>
				<tr>
					<th scope="row">自动特色图（可选）</th>
					<td>
						<label>
							<input type="checkbox" id="wp-aiw-writer-url-auto-featured" value="1" />
							生成预览图 → 下载入库 → 设为特色图（失败不影响草稿）
						</label>
						<p class="description">图片提示词仅使用标题。需在设置页开启 image_enabled。</p>
					</td>
				</tr>
			</table>

			<p>
				<button id="wp-aiw-writer-url-generate" class="button button-primary">抓取并生成草稿</button>
				<span id="wp-aiw-writer-url-status" style="margin-left: 10px;"></span>
			</p>

			<div id="wp-aiw-writer-url-result" style="margin-top: 12px;"></div>
			<div id="wp-aiw-writer-url-log" style="margin-top: 12px; padding: 12px; background: #fff; border: 1px solid #dcdcde; border-radius: 6px; white-space: pre-wrap;"></div>
		</div>
		<?php
	}

	public function render_writer_batch_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$default_lang = 'zh';
		?>
		<div class="wrap">
			<h1>AI 批量写文章</h1>
			<p>每行一个任务，按队列批量生成草稿。支持本地存储、暂停/继续、刷新后恢复未完成队列。</p>

			<div id="wp-aiw-writer-batch-input">
				<table class="form-table" role="presentation">
					<tr>
						<th scope="row"><label for="wp-aiw-writer-batch-titles">任务（每行一个）</label></th>
						<td>
							<textarea id="wp-aiw-writer-batch-titles" class="large-text" rows="10" placeholder="格式：标题 || url1,url2 || 参考内容\n\n示例：\n如何用 WordPress 做 SaaS 多租户 || https://example.com/a, https://example.com/b || 请突出多租户数据隔离\nOpenAI-compatible 接口最佳实践 || || 重点写鉴权/超时/重试\n仅标题也可以\n"></textarea>
							<p class="description">分隔符使用 <code>||</code>。URL 可用英文逗号/中文逗号分隔；参考内容可留空。</p>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="wp-aiw-writer-batch-req">写作要求（可选）</label></th>
						<td>
							<textarea id="wp-aiw-writer-batch-req" class="large-text" rows="6" placeholder="例如：面向初学者，分步骤，给出常见错误排查。"></textarea>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="wp-aiw-writer-batch-lang">语言</label></th>
						<td>
							<select id="wp-aiw-writer-batch-lang">
								<option value="zh" <?php selected( $default_lang, 'zh' ); ?>>中文</option>
								<option value="en" <?php selected( $default_lang, 'en' ); ?>>English</option>
							</select>
							<label style="margin-left: 12px;">
								<input type="checkbox" id="wp-aiw-writer-batch-debug" value="1" /> 返回调试信息
							</label>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="wp-aiw-writer-batch-concurrency">并发（本页专用）</label></th>
						<td>
							<input id="wp-aiw-writer-batch-concurrency" type="number" min="1" max="10" class="small-text" />
							<p class="description">与 runner 的 Batch Size（批量重写并发）区分。默认读取设置项 writer_batch_size。</p>
						</td>
					</tr>
					<tr>
						<th scope="row">自动特色图（可选）</th>
						<td>
							<label>
								<input type="checkbox" id="wp-aiw-writer-batch-auto-featured" value="1" />
								每篇自动生成预览图 → 下载入库 → 设为特色图（失败不影响草稿）
							</label>
							<p class="description">图片提示词仅使用标题 + tags。需在设置页开启 image_enabled。</p>
						</td>
					</tr>
				</table>

				<p>
					<button id="wp-aiw-writer-batch-start" class="button button-primary">开始批量生成</button>
					<button id="wp-aiw-writer-batch-clear" class="button" style="margin-left:8px;">清除未完成任务</button>
					<span id="wp-aiw-writer-batch-status" style="margin-left:10px;"></span>
				</p>
			</div>

			<div id="wp-aiw-writer-batch-progress" style="display:none;">
				<p>
					<button id="wp-aiw-writer-batch-pause" class="button">暂停</button>
					<button id="wp-aiw-writer-batch-resume" class="button button-primary" style="margin-left:8px;">继续</button>
					<button id="wp-aiw-writer-batch-retry-failed" class="button" style="margin-left:8px;">重试全部失败项</button>
					<button id="wp-aiw-writer-batch-retry-selected" class="button" style="margin-left:8px;">重试选中项</button>
					<button id="wp-aiw-writer-batch-delete-selected-failed" class="button" style="margin-left:8px;">删除选中失败项</button>
					<button id="wp-aiw-writer-batch-clear2" class="button" style="margin-left:8px;">清除未完成任务</button>
					<span id="wp-aiw-writer-batch-status2" style="margin-left:10px;"></span>
				</p>
				<div id="wp-aiw-writer-batch-list" style="margin-top:12px; padding:12px; background:#fff; border:1px solid #dcdcde; border-radius:6px;"></div>
			</div>
		</div>
		<?php
	}

	public function ajax_generate_draft_from_urls() {
		if ( ! current_user_can( 'edit_posts' ) ) {
			wp_send_json_error( array( 'message' => '无权限' ), 403 );
		}
		check_ajax_referer( 'wp_aiw_nonce', 'nonce' );

		$requirements = isset( $_POST['requirements'] ) ? wp_unslash( (string) $_POST['requirements'] ) : '';
		$requirements = is_string( $requirements ) ? trim( $requirements ) : '';

		$reference_text = isset( $_POST['referenceText'] ) ? wp_unslash( (string) $_POST['referenceText'] ) : '';
		$reference_text = is_string( $reference_text ) ? trim( $reference_text ) : '';

		$language = isset( $_POST['language'] ) ? sanitize_key( wp_unslash( $_POST['language'] ) ) : 'zh';
		if ( ! in_array( $language, array( 'zh', 'en' ), true ) ) {
			$language = 'zh';
		}

		$urls_raw = isset( $_POST['urls'] ) ? wp_unslash( (string) $_POST['urls'] ) : '';
		$urls = $this->split_urls( $urls_raw );
		if ( count( $urls ) > 3 ) {
			$urls = array_slice( $urls, 0, 3 );
		}

		$writer = WP_AIW_Writer::instance();
		$r = $writer->generate_draft_from_urls(
			array(
				'requirements' => $requirements,
				'reference_text' => $reference_text,
				'language' => $language,
				'urls' => $urls,
			)
		);

		if ( empty( $r['ok'] ) ) {
			wp_send_json_error( array( 'message' => isset( $r['error'] ) ? (string) $r['error'] : '生成失败', 'debug' => isset( $r['debug'] ) ? $r['debug'] : null ), 500 );
		}

		$payload = array(
			'postId' => isset( $r['post_id'] ) ? (int) $r['post_id'] : 0,
			'editUrl' => isset( $r['edit_url'] ) ? (string) $r['edit_url'] : '',
			'title' => isset( $r['title'] ) ? (string) $r['title'] : '',
			'tags' => isset( $r['tags'] ) ? $r['tags'] : array(),
		);

		// Always return URL fetch errors (if any) as warnings for better UX.
		if ( isset( $r['debug'] ) && is_array( $r['debug'] ) && ! empty( $r['debug']['url_errors'] ) && is_array( $r['debug']['url_errors'] ) ) {
			$payload['urlErrors'] = $r['debug']['url_errors'];
		}

		$want_debug = ! empty( $_POST['debug'] );
		if ( $want_debug && isset( $r['debug'] ) ) {
			$payload['debug'] = $r['debug'];
		}

		wp_send_json_success( $payload );
	}

	public function ajax_generate_draft() {
		if ( ! current_user_can( 'edit_posts' ) ) {
			wp_send_json_error( array( 'message' => '无权限' ), 403 );
		}
		check_ajax_referer( 'wp_aiw_nonce', 'nonce' );

		$title = isset( $_POST['title'] ) ? sanitize_text_field( wp_unslash( $_POST['title'] ) ) : '';
		$requirements = isset( $_POST['requirements'] ) ? wp_unslash( (string) $_POST['requirements'] ) : '';
		$requirements = is_string( $requirements ) ? trim( $requirements ) : '';
		$reference_text = isset( $_POST['referenceText'] ) ? wp_unslash( (string) $_POST['referenceText'] ) : '';
		$reference_text = is_string( $reference_text ) ? trim( $reference_text ) : '';

		$language = isset( $_POST['language'] ) ? sanitize_key( wp_unslash( $_POST['language'] ) ) : 'zh';
		if ( ! in_array( $language, array( 'zh', 'en' ), true ) ) {
			$language = 'zh';
		}

		$urls_raw = isset( $_POST['urls'] ) ? wp_unslash( (string) $_POST['urls'] ) : '';
		$urls = $this->split_urls( $urls_raw );

		$writer = WP_AIW_Writer::instance();
		$r = $writer->generate_draft(
			array(
				'title' => $title,
				'requirements' => $requirements,
				'language' => $language,
				'urls' => $urls,
				'reference_text' => $reference_text,
			)
		);

		if ( empty( $r['ok'] ) ) {
			wp_send_json_error( array( 'message' => isset( $r['error'] ) ? (string) $r['error'] : '生成失败', 'debug' => isset( $r['debug'] ) ? $r['debug'] : null ), 500 );
		}

		$payload = array(
			'postId' => isset( $r['post_id'] ) ? (int) $r['post_id'] : 0,
			'editUrl' => isset( $r['edit_url'] ) ? (string) $r['edit_url'] : '',
			'title' => isset( $r['title'] ) ? (string) $r['title'] : '',
			'tags' => isset( $r['tags'] ) ? $r['tags'] : array(),
		);

		$want_debug = ! empty( $_POST['debug'] );
		if ( $want_debug && isset( $r['debug'] ) ) {
			$payload['debug'] = $r['debug'];
		}

		wp_send_json_success( $payload );
	}

	public function ajax_generate_content() {
		if ( ! current_user_can( 'edit_posts' ) ) {
			wp_send_json_error( array( 'message' => '无权限' ), 403 );
		}
		check_ajax_referer( 'wp_aiw_nonce', 'nonce' );

		$title = isset( $_POST['title'] ) ? sanitize_text_field( wp_unslash( $_POST['title'] ) ) : '';
		$requirements = isset( $_POST['requirements'] ) ? wp_unslash( (string) $_POST['requirements'] ) : '';
		$requirements = is_string( $requirements ) ? trim( $requirements ) : '';
		$reference_text = isset( $_POST['referenceText'] ) ? wp_unslash( (string) $_POST['referenceText'] ) : '';
		$reference_text = is_string( $reference_text ) ? trim( $reference_text ) : '';

		$language = isset( $_POST['language'] ) ? sanitize_key( wp_unslash( $_POST['language'] ) ) : 'zh';
		if ( ! in_array( $language, array( 'zh', 'en' ), true ) ) {
			$language = 'zh';
		}

		$urls_raw = isset( $_POST['urls'] ) ? wp_unslash( (string) $_POST['urls'] ) : '';
		$urls = $this->split_urls( $urls_raw );

		$writer = WP_AIW_Writer::instance();
		$r = $writer->generate_content(
			array(
				'title' => $title,
				'requirements' => $requirements,
				'language' => $language,
				'urls' => $urls,
				'reference_text' => $reference_text,
			)
		);

		if ( empty( $r['ok'] ) ) {
			wp_send_json_error( array( 'message' => isset( $r['error'] ) ? (string) $r['error'] : '生成失败', 'debug' => isset( $r['debug'] ) ? $r['debug'] : null ), 500 );
		}

		$payload = array(
			'title' => isset( $r['title'] ) ? (string) $r['title'] : '',
			'contentHtml' => isset( $r['content_html'] ) ? (string) $r['content_html'] : '',
			'tags' => isset( $r['tags'] ) ? $r['tags'] : array(),
		);

		$want_debug = ! empty( $_POST['debug'] );
		if ( $want_debug && isset( $r['debug'] ) ) {
			$payload['debug'] = $r['debug'];
		}

		wp_send_json_success( $payload );
	}

	public function ajax_generate_rewrite_current() {
		if ( ! current_user_can( 'edit_posts' ) ) {
			wp_send_json_error( array( 'message' => '无权限' ), 403 );
		}
		check_ajax_referer( 'wp_aiw_nonce', 'nonce' );

		$post_id = isset( $_POST['postId'] ) ? (int) $_POST['postId'] : 0;
		if ( $post_id <= 0 ) {
			wp_send_json_error( array( 'message' => 'postId 无效' ), 400 );
		}
		if ( ! current_user_can( 'edit_post', $post_id ) ) {
			wp_send_json_error( array( 'message' => '无权限编辑该文章' ), 403 );
		}

		$title = isset( $_POST['title'] ) ? sanitize_text_field( wp_unslash( $_POST['title'] ) ) : '';
		$content = isset( $_POST['content'] ) ? wp_unslash( (string) $_POST['content'] ) : '';
		$content = is_string( $content ) ? (string) $content : '';

		$requirements = isset( $_POST['requirements'] ) ? wp_unslash( (string) $_POST['requirements'] ) : '';
		$requirements = is_string( $requirements ) ? trim( $requirements ) : '';
		$reference_text = isset( $_POST['referenceText'] ) ? wp_unslash( (string) $_POST['referenceText'] ) : '';
		$reference_text = is_string( $reference_text ) ? trim( $reference_text ) : '';

		$urls_raw = isset( $_POST['urls'] ) ? wp_unslash( (string) $_POST['urls'] ) : '';
		$urls = $this->split_urls( $urls_raw );

		$language = isset( $_POST['language'] ) ? sanitize_key( wp_unslash( $_POST['language'] ) ) : 'zh';
		if ( ! in_array( $language, array( 'zh', 'en' ), true ) ) {
			$language = 'zh';
		}

		$rewrite_title = isset( $_POST['rewriteTitle'] ) ? (bool) (int) $_POST['rewriteTitle'] : false;

		$settings = WP_AIW_Settings::instance()->get();
		$rewriter = new WP_AIW_Rewriter();
		$r = $rewriter->generate_rewrite(
			$title,
			$content,
			$settings,
			array(
				'language' => $language,
				'rewrite_title' => $rewrite_title,
				'requirements' => $requirements,
				'reference_text' => $reference_text,
				'urls' => $urls,
			)
		);

		if ( empty( $r['ok'] ) ) {
			$payload = array(
				'message' => isset( $r['error'] ) ? (string) $r['error'] : '生成失败',
			);

			$want_debug = ! empty( $_POST['debug'] );
			if ( $want_debug && isset( $r['raw'] ) ) {
				$payload['debug'] = array( 'raw' => $r['raw'] );
			}

			wp_send_json_error( $payload, 500 );
		}

		$payload = array(
			'contentHtml' => isset( $r['content_html'] ) ? (string) $r['content_html'] : '',
			'tags' => isset( $r['tags'] ) && is_array( $r['tags'] ) ? $r['tags'] : array(),
			'title' => isset( $r['title'] ) ? (string) $r['title'] : '',
		);

		$want_debug = ! empty( $_POST['debug'] );
		if ( $want_debug ) {
			$payload['debug'] = array();
			if ( isset( $r['raw'] ) ) {
				$payload['debug']['raw'] = $r['raw'];
			}
			if ( isset( $r['debug'] ) && is_array( $r['debug'] ) ) {
				$payload['debug'] = array_merge( $payload['debug'], $r['debug'] );
			}
			if ( empty( $payload['debug'] ) ) {
				unset( $payload['debug'] );
			}
		}

		wp_send_json_success( $payload );
	}

	private function split_urls( $raw ) {
		$raw = is_string( $raw ) ? trim( $raw ) : '';
		if ( $raw === '' ) {
			return array();
		}
		$lines = preg_split( '/\r\n|\r|\n/', $raw );
		if ( ! is_array( $lines ) ) {
			return array();
		}
		$urls = array();
		foreach ( $lines as $line ) {
			$line = trim( (string) $line );
			if ( $line === '' ) {
				continue;
			}
			$urls[] = $line;
		}
		$urls = array_values( array_unique( $urls ) );
		return $urls;
	}
}
