<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class WP_AIW_Settings {
	private static $instance = null;

	public static function instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	public function get_option_name() {
		return 'wp_aiw_settings';
	}

	public function get_defaults() {
		$default_prompt_zh = $this->get_default_prompt_zh();
		$default_prompt_en = $this->get_default_prompt_en();
		$default_writer_prompt_zh = $this->get_default_writer_prompt_zh();
		$default_writer_prompt_en = $this->get_default_writer_prompt_en();
		$default_image_prompt_prefix_zh = $this->get_default_image_prompt_prefix_zh();
		$default_image_prompt_prefix_en = $this->get_default_image_prompt_prefix_en();
		return array(
			'llm_base_url' => 'https://api.deepseek.com',
			'llm_api_key'  => '',
			'llm_model'    => 'deepseek-chat',
			'llm_timeout_seconds' => 180,
			'llm_max_tokens' => 8000,
			'batch_size'   => 1,
			'writer_batch_size' => 2,
			'prompt_system_zh' => $default_prompt_zh,
			'prompt_system_en' => $default_prompt_en,

			// Writer (AI 写文章) - Phase1
			'writer_prompt_system_zh' => $default_writer_prompt_zh,
			'writer_prompt_system_en' => $default_writer_prompt_en,
			'writer_summary_first' => 1,
			'url_fetch_timeout_seconds' => 30,
			'url_fetch_max_chars' => 12000,
			'url_fetch_max_urls' => 3,

			// Image generation (Phase3, decoupled from text LLM)
			'image_enabled' => 0,
			'image_provider' => 'zhipu_glm_image',
			'image_base_url' => 'https://open.bigmodel.cn/api/paas/v4',
			'image_api_key' => '',
			'image_model' => 'glm-image',
			'image_timeout_seconds' => 180,
			'image_size' => '1280x1280',
			'image_response_format' => 'b64_json',
			'image_extra_params_json' => '',
			'image_max_per_post' => 1,
			'image_prompt_prefix_zh' => $default_image_prompt_prefix_zh,
			'image_prompt_prefix_en' => $default_image_prompt_prefix_en,
		);
	}

	private function get_default_image_prompt_prefix_zh() {
		return '你是专业的技术文章插图提示词生成器。请生成适合配图的英文 prompt（简洁、具象、避免文字/水印），风格偏现代扁平/科技感插画。';
	}

	private function get_default_image_prompt_prefix_en() {
		return 'You generate concise English prompts for technical article illustrations. Avoid text/watermarks. Use a modern flat tech illustration style.';
	}

	private function get_default_writer_prompt_zh() {
		return <<<'PROMPT'
你是中文技术写作编辑。请根据用户给出的“文章标题/写作要求”和“参考资料（可选，可能来自 URL 抓取摘要/正文）”，生成一篇适合 WordPress 发布的技术教程文章。

要求：
1) 输出必须是严格 JSON（不要 Markdown 代码块）。
	- 请自行控制输出长度，确保在你可稳定输出的最大范围内完成；宁可精简内容，也不要因过长导致输出被截断。
	- 你必须保证最终输出是一个完整且可解析的 JSON 对象：所有花括号/方括号/引号必须闭合；禁止输出半截 JSON。
	- 如果预计内容过长：请优先压缩 content_html（减少段落、缩短示例代码、合并列表/表格、去掉非关键解释），但仍需保持结构完整与可读性。
2) JSON 字段：title（string, 必填）, content_html（string, 必填）, tags（array of string, 可选）。
3) content_html 必须是可直接写入 WordPress 的 HTML（允许 <h2><h3><p><ul><ol><li><blockquote><pre><code><table> 等）。
4) 标题层级：不要输出 <h1>（H1 由文章标题承担）。正文从 <h2> 开始，必要时使用 <h3>，不要跳级。
5) 文章风格：技术教程、结构清晰、步骤可执行；必要处给出代码块与注意事项。
6) 如果参考资料存在冲突/不完整，以更稳妥的通用做法为准，并明确说明假设。
7) tags：给出 3~8 个相关短标签（中英文均可，不含#），避免无关词。
PROMPT;
	}

	private function get_default_writer_prompt_en() {
		return <<<'PROMPT'
You are an English technical writer. Using the given “title / requirements” and “references (optional, may come from URL fetched summaries/content)”, write a WordPress-ready technical tutorial.

Requirements:
1) Output MUST be strict JSON (no Markdown fences).
	- Control your output length so you can reliably finish within your safe output limit; prefer brevity over being cut off due to excessive length.
	- You MUST ensure the final output is a complete, parseable JSON object: all braces/brackets/quotes must be properly closed; never output partial JSON.
	- If the output might be too long: prioritize compressing content_html (fewer paragraphs, shorter code samples, merge lists/tables, remove non-essential explanations) while keeping structure and readability.
2) JSON fields: title (string, required), content_html (string, required), tags (array of string, optional).
3) content_html must be HTML suitable for WordPress (you may use <h2><h3><p><ul><ol><li><blockquote><pre><code><table>, etc.).
4) Heading hierarchy: do NOT output <h1> (H1 is the post title). Start from <h2>, then use <h3> when needed. Do not skip levels.
5) Style: practical, step-by-step, with runnable snippets and cautions.
6) If references are incomplete or conflicting, prefer safe general practices and state assumptions.
7) tags: provide 3-8 short relevant tags, without '#'.
PROMPT;
	}

	private function get_default_prompt_zh() {
		return <<<'PROMPT'
你是中文技术写作编辑。你的任务是把给定的 WordPress 文章内容重写为更标准、更易读、更适配 SaaS 主题正文渲染的 HTML 内容，同时修正错误并补充必要说明。

要求：
1) 输出必须是严格 JSON（不要 Markdown 代码块）。
	- 请自行控制输出长度，确保在你可稳定输出的最大范围内完成；宁可精简内容，也不要因过长导致输出被截断。
	- 你必须保证最终输出是一个完整且可解析的 JSON 对象：所有花括号/方括号/引号必须闭合；禁止输出半截 JSON。
	- 如果预计内容过长：请优先压缩 content_html（减少段落、缩短示例代码、合并列表/表格、去掉非关键解释），但仍需保持结构完整与可读性。
2) JSON 字段：content_html（string, 必填）, tags（array of string, 可选）, title（string, 可选）。
3) content_html 必须是可直接写入 WordPress 的 HTML（允许 <h2><h3><p><ul><ol><li><blockquote><pre><code><table> 等）。
4) 标题层级：不要输出 <h1>（H1 由文章标题承担）。正文从 <h2> 开始，必要时使用 <h3>，不要跳级。
5) 保留代码缩进：代码请放在 <pre><code> 中，且代码内容不要被解释执行。
6) 如果原文存在明显错误/不完整，修正并补充更标准的写法与说明。
7) tags：给出 3~8 个中文或英文短标签（不含#），避免与无关词。
PROMPT;
	}

	private function get_default_prompt_en() {
		return <<<'PROMPT'
You are an English technical editor. Rewrite the given WordPress post into clear, well-structured, theme-friendly HTML content. Fix mistakes and fill in missing explanations where necessary.

Requirements:
1) Output MUST be strict JSON (no Markdown fences).
	- Control your output length so you can reliably finish within your safe output limit; prefer brevity over being cut off due to excessive length.
	- You MUST ensure the final output is a complete, parseable JSON object: all braces/brackets/quotes must be properly closed; never output partial JSON.
	- If the output might be too long: prioritize compressing content_html (fewer paragraphs, shorter code samples, merge lists/tables, remove non-essential explanations) while keeping structure and readability.
2) JSON fields: content_html (string, required), tags (array of string, optional), title (string, optional).
3) content_html must be HTML suitable for WordPress (you may use <h2><h3><p><ul><ol><li><blockquote><pre><code><table>, etc.).
4) Heading hierarchy: do NOT output <h1> (H1 is the post title). Start from <h2>, then use <h3> when needed. Do not skip levels.
5) Preserve code indentation: put code inside <pre><code> and do not execute/interpret it.
6) Correct errors and improve completeness with proper explanations.
7) tags: provide 3-8 short tags (English), without '#', relevant only.
PROMPT;
	}

	public function get() {
		$defaults = $this->get_defaults();
		$current  = get_option( $this->get_option_name(), array() );
		if ( ! is_array( $current ) ) {
			$current = array();
		}
		$merged = array_merge( $defaults, $current );
		$merged['batch_size'] = max( 1, (int) $merged['batch_size'] );
		$merged['writer_batch_size'] = isset( $merged['writer_batch_size'] ) ? (int) $merged['writer_batch_size'] : (int) $defaults['writer_batch_size'];
		$merged['writer_batch_size'] = max( 1, min( 10, (int) $merged['writer_batch_size'] ) );
		$merged['llm_timeout_seconds'] = isset( $merged['llm_timeout_seconds'] ) ? (int) $merged['llm_timeout_seconds'] : (int) $defaults['llm_timeout_seconds'];
		$merged['llm_timeout_seconds'] = max( 5, min( 600, (int) $merged['llm_timeout_seconds'] ) );

		$merged['llm_max_tokens'] = isset( $merged['llm_max_tokens'] ) ? (int) $merged['llm_max_tokens'] : (int) $defaults['llm_max_tokens'];
		// 0 means "use provider/model default".
		$merged['llm_max_tokens'] = max( 0, min( 65536, (int) $merged['llm_max_tokens'] ) );
		$merged['llm_base_url'] = is_string( $merged['llm_base_url'] ) ? trim( $merged['llm_base_url'] ) : $defaults['llm_base_url'];
		$merged['llm_api_key']  = is_string( $merged['llm_api_key'] ) ? trim( $merged['llm_api_key'] ) : '';
		$merged['llm_model']    = is_string( $merged['llm_model'] ) ? trim( $merged['llm_model'] ) : $defaults['llm_model'];
		$merged['prompt_system_zh'] = is_string( $merged['prompt_system_zh'] ) ? (string) $merged['prompt_system_zh'] : $defaults['prompt_system_zh'];
		$merged['prompt_system_en'] = is_string( $merged['prompt_system_en'] ) ? (string) $merged['prompt_system_en'] : $defaults['prompt_system_en'];

		$merged['writer_prompt_system_zh'] = is_string( $merged['writer_prompt_system_zh'] ) ? (string) $merged['writer_prompt_system_zh'] : $defaults['writer_prompt_system_zh'];
		$merged['writer_prompt_system_en'] = is_string( $merged['writer_prompt_system_en'] ) ? (string) $merged['writer_prompt_system_en'] : $defaults['writer_prompt_system_en'];
		$merged['writer_prompt_system_zh'] = trim( $merged['writer_prompt_system_zh'] );
		$merged['writer_prompt_system_en'] = trim( $merged['writer_prompt_system_en'] );
		if ( $merged['writer_prompt_system_zh'] === '' ) {
			$merged['writer_prompt_system_zh'] = $defaults['writer_prompt_system_zh'];
		}
		if ( $merged['writer_prompt_system_en'] === '' ) {
			$merged['writer_prompt_system_en'] = $defaults['writer_prompt_system_en'];
		}
		$merged['writer_summary_first'] = ! empty( $merged['writer_summary_first'] ) ? 1 : 0;
		$merged['url_fetch_timeout_seconds'] = isset( $merged['url_fetch_timeout_seconds'] ) ? (int) $merged['url_fetch_timeout_seconds'] : (int) $defaults['url_fetch_timeout_seconds'];
		$merged['url_fetch_timeout_seconds'] = max( 3, min( 60, (int) $merged['url_fetch_timeout_seconds'] ) );
		$merged['url_fetch_max_chars'] = isset( $merged['url_fetch_max_chars'] ) ? (int) $merged['url_fetch_max_chars'] : (int) $defaults['url_fetch_max_chars'];
		$merged['url_fetch_max_chars'] = max( 1000, min( 100000, (int) $merged['url_fetch_max_chars'] ) );
		$merged['url_fetch_max_urls'] = isset( $merged['url_fetch_max_urls'] ) ? (int) $merged['url_fetch_max_urls'] : (int) $defaults['url_fetch_max_urls'];
		$merged['url_fetch_max_urls'] = max( 1, min( 10, (int) $merged['url_fetch_max_urls'] ) );

		$merged['image_enabled'] = ! empty( $merged['image_enabled'] ) ? 1 : 0;
		$merged['image_provider'] = isset( $merged['image_provider'] ) ? trim( (string) $merged['image_provider'] ) : (string) $defaults['image_provider'];
		$allowed_provider = array( 'zhipu_glm_image', 'gemini_generatecontent', 'openai_compatible_images' );
		if ( ! in_array( $merged['image_provider'], $allowed_provider, true ) ) {
			$merged['image_provider'] = (string) $defaults['image_provider'];
		}
		$merged['image_base_url'] = isset( $merged['image_base_url'] ) ? rtrim( trim( (string) $merged['image_base_url'] ), '/' ) : '';
		$merged['image_api_key'] = isset( $merged['image_api_key'] ) ? trim( (string) $merged['image_api_key'] ) : '';
		$merged['image_model'] = isset( $merged['image_model'] ) ? trim( (string) $merged['image_model'] ) : '';
		$merged['image_timeout_seconds'] = isset( $merged['image_timeout_seconds'] ) ? (int) $merged['image_timeout_seconds'] : (int) $defaults['image_timeout_seconds'];
		$merged['image_timeout_seconds'] = max( 10, min( 600, (int) $merged['image_timeout_seconds'] ) );
		$merged['image_size'] = isset( $merged['image_size'] ) ? trim( (string) $merged['image_size'] ) : (string) $defaults['image_size'];
		if ( $merged['image_size'] === '' ) {
			$merged['image_size'] = (string) $defaults['image_size'];
		}
		$merged['image_response_format'] = isset( $merged['image_response_format'] ) ? trim( (string) $merged['image_response_format'] ) : (string) $defaults['image_response_format'];
		if ( ! in_array( $merged['image_response_format'], array( 'b64_json', 'url' ), true ) ) {
			$merged['image_response_format'] = (string) $defaults['image_response_format'];
		}
		$merged['image_extra_params_json'] = isset( $merged['image_extra_params_json'] ) ? trim( (string) $merged['image_extra_params_json'] ) : '';
		$merged['image_max_per_post'] = isset( $merged['image_max_per_post'] ) ? (int) $merged['image_max_per_post'] : (int) $defaults['image_max_per_post'];
		$merged['image_max_per_post'] = max( 0, min( 20, (int) $merged['image_max_per_post'] ) );
		$merged['image_prompt_prefix_zh'] = isset( $merged['image_prompt_prefix_zh'] ) ? trim( (string) $merged['image_prompt_prefix_zh'] ) : (string) $defaults['image_prompt_prefix_zh'];
		$merged['image_prompt_prefix_en'] = isset( $merged['image_prompt_prefix_en'] ) ? trim( (string) $merged['image_prompt_prefix_en'] ) : (string) $defaults['image_prompt_prefix_en'];
		if ( $merged['image_prompt_prefix_zh'] === '' ) {
			$merged['image_prompt_prefix_zh'] = (string) $defaults['image_prompt_prefix_zh'];
		}
		if ( $merged['image_prompt_prefix_en'] === '' ) {
			$merged['image_prompt_prefix_en'] = (string) $defaults['image_prompt_prefix_en'];
		}
		return $merged;
	}

	public function sanitize( $input ) {
		$defaults = $this->get_defaults();
		$out = array();
		$out['llm_base_url'] = isset( $input['llm_base_url'] ) ? trim( (string) $input['llm_base_url'] ) : $defaults['llm_base_url'];
		$out['llm_api_key']  = isset( $input['llm_api_key'] ) ? trim( (string) $input['llm_api_key'] ) : '';
		$out['llm_model']    = isset( $input['llm_model'] ) ? trim( (string) $input['llm_model'] ) : $defaults['llm_model'];
		$out['llm_timeout_seconds'] = isset( $input['llm_timeout_seconds'] ) ? (int) $input['llm_timeout_seconds'] : (int) $defaults['llm_timeout_seconds'];
		$out['llm_timeout_seconds'] = max( 5, min( 600, (int) $out['llm_timeout_seconds'] ) );
		$out['llm_max_tokens'] = isset( $input['llm_max_tokens'] ) ? (int) $input['llm_max_tokens'] : (int) $defaults['llm_max_tokens'];
		$out['llm_max_tokens'] = max( 0, min( 65536, (int) $out['llm_max_tokens'] ) );
		$out['batch_size']   = isset( $input['batch_size'] ) ? max( 1, (int) $input['batch_size'] ) : 1;
		$out['writer_batch_size'] = isset( $input['writer_batch_size'] ) ? (int) $input['writer_batch_size'] : (int) $defaults['writer_batch_size'];
		$out['writer_batch_size'] = max( 1, min( 10, (int) $out['writer_batch_size'] ) );
		$out['prompt_system_zh'] = isset( $input['prompt_system_zh'] ) ? (string) $input['prompt_system_zh'] : $defaults['prompt_system_zh'];
		$out['prompt_system_en'] = isset( $input['prompt_system_en'] ) ? (string) $input['prompt_system_en'] : $defaults['prompt_system_en'];
		$out['writer_prompt_system_zh'] = isset( $input['writer_prompt_system_zh'] ) ? (string) $input['writer_prompt_system_zh'] : $defaults['writer_prompt_system_zh'];
		$out['writer_prompt_system_en'] = isset( $input['writer_prompt_system_en'] ) ? (string) $input['writer_prompt_system_en'] : $defaults['writer_prompt_system_en'];
		$out['writer_summary_first'] = ! empty( $input['writer_summary_first'] ) ? 1 : 0;
		$out['url_fetch_timeout_seconds'] = isset( $input['url_fetch_timeout_seconds'] ) ? (int) $input['url_fetch_timeout_seconds'] : (int) $defaults['url_fetch_timeout_seconds'];
		$out['url_fetch_timeout_seconds'] = max( 3, min( 60, (int) $out['url_fetch_timeout_seconds'] ) );
		$out['url_fetch_max_chars'] = isset( $input['url_fetch_max_chars'] ) ? (int) $input['url_fetch_max_chars'] : (int) $defaults['url_fetch_max_chars'];
		$out['url_fetch_max_chars'] = max( 1000, min( 100000, (int) $out['url_fetch_max_chars'] ) );
		$out['url_fetch_max_urls'] = isset( $input['url_fetch_max_urls'] ) ? (int) $input['url_fetch_max_urls'] : (int) $defaults['url_fetch_max_urls'];
		$out['url_fetch_max_urls'] = max( 1, min( 10, (int) $out['url_fetch_max_urls'] ) );

		$out['image_enabled'] = ! empty( $input['image_enabled'] ) ? 1 : 0;
		$out['image_provider'] = isset( $input['image_provider'] ) ? trim( (string) $input['image_provider'] ) : (string) $defaults['image_provider'];
		$allowed_provider = array( 'zhipu_glm_image', 'gemini_generatecontent', 'openai_compatible_images' );
		if ( ! in_array( $out['image_provider'], $allowed_provider, true ) ) {
			$out['image_provider'] = (string) $defaults['image_provider'];
		}
		$out['image_base_url'] = isset( $input['image_base_url'] ) ? rtrim( trim( (string) $input['image_base_url'] ), '/' ) : '';
		$out['image_api_key'] = isset( $input['image_api_key'] ) ? trim( (string) $input['image_api_key'] ) : '';
		$out['image_model'] = isset( $input['image_model'] ) ? trim( (string) $input['image_model'] ) : '';
		$out['image_timeout_seconds'] = isset( $input['image_timeout_seconds'] ) ? (int) $input['image_timeout_seconds'] : (int) $defaults['image_timeout_seconds'];
		$out['image_timeout_seconds'] = max( 10, min( 600, (int) $out['image_timeout_seconds'] ) );
		$out['image_size'] = isset( $input['image_size'] ) ? trim( (string) $input['image_size'] ) : (string) $defaults['image_size'];
		if ( $out['image_size'] === '' ) {
			$out['image_size'] = (string) $defaults['image_size'];
		}
		$out['image_response_format'] = isset( $input['image_response_format'] ) ? trim( (string) $input['image_response_format'] ) : (string) $defaults['image_response_format'];
		if ( ! in_array( $out['image_response_format'], array( 'b64_json', 'url' ), true ) ) {
			$out['image_response_format'] = (string) $defaults['image_response_format'];
		}
		$out['image_extra_params_json'] = isset( $input['image_extra_params_json'] ) ? trim( (string) $input['image_extra_params_json'] ) : '';
		$out['image_max_per_post'] = isset( $input['image_max_per_post'] ) ? (int) $input['image_max_per_post'] : (int) $defaults['image_max_per_post'];
		$out['image_max_per_post'] = max( 0, min( 20, (int) $out['image_max_per_post'] ) );
		$out['image_prompt_prefix_zh'] = isset( $input['image_prompt_prefix_zh'] ) ? trim( (string) $input['image_prompt_prefix_zh'] ) : (string) $defaults['image_prompt_prefix_zh'];
		$out['image_prompt_prefix_en'] = isset( $input['image_prompt_prefix_en'] ) ? trim( (string) $input['image_prompt_prefix_en'] ) : (string) $defaults['image_prompt_prefix_en'];
		if ( $out['image_prompt_prefix_zh'] === '' ) {
			$out['image_prompt_prefix_zh'] = (string) $defaults['image_prompt_prefix_zh'];
		}
		if ( $out['image_prompt_prefix_en'] === '' ) {
			$out['image_prompt_prefix_en'] = (string) $defaults['image_prompt_prefix_en'];
		}

		// Basic normalization
		if ( $out['llm_base_url'] === '' ) {
			$out['llm_base_url'] = $defaults['llm_base_url'];
		}
		$out['llm_base_url'] = rtrim( $out['llm_base_url'], '/' );
		$out['prompt_system_zh'] = trim( $out['prompt_system_zh'] );
		$out['prompt_system_en'] = trim( $out['prompt_system_en'] );
		$out['writer_prompt_system_zh'] = trim( $out['writer_prompt_system_zh'] );
		$out['writer_prompt_system_en'] = trim( $out['writer_prompt_system_en'] );
		$out['image_prompt_prefix_zh'] = trim( $out['image_prompt_prefix_zh'] );
		$out['image_prompt_prefix_en'] = trim( $out['image_prompt_prefix_en'] );
		if ( $out['prompt_system_zh'] === '' ) {
			$out['prompt_system_zh'] = $defaults['prompt_system_zh'];
		}
		if ( $out['prompt_system_en'] === '' ) {
			$out['prompt_system_en'] = $defaults['prompt_system_en'];
		}
		if ( $out['writer_prompt_system_zh'] === '' ) {
			$out['writer_prompt_system_zh'] = $defaults['writer_prompt_system_zh'];
		}
		if ( $out['writer_prompt_system_en'] === '' ) {
			$out['writer_prompt_system_en'] = $defaults['writer_prompt_system_en'];
		}

		return $out;
	}

	private function __construct() {
		add_action( 'admin_init', array( $this, 'register_settings' ) );
	}

	public function register_settings() {
		register_setting(
			'wp_aiw_settings_group',
			$this->get_option_name(),
			array( 'sanitize_callback' => array( $this, 'sanitize' ) )
		);
	}
}
