<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class WP_AIW_LLM_Client {
	/**
	 * Calls an OpenAI-compatible Chat Completions endpoint.
	 *
	 * @return array{ok:bool, data?:array, error?:string, raw?:mixed}
	 */
	public function chat_completions( $base_url, $api_key, $model, $messages, $timeout = 60, $max_tokens = 0 ) {
		$base_url = is_string( $base_url ) ? rtrim( trim( $base_url ), '/' ) : '';
		$api_key  = is_string( $api_key ) ? trim( $api_key ) : '';
		$model    = is_string( $model ) ? trim( $model ) : '';

		if ( $base_url === '' ) {
			return array( 'ok' => false, 'error' => 'llm_base_url 不能为空' );
		}
		if ( $api_key === '' ) {
			return array( 'ok' => false, 'error' => 'llm_api_key 未配置' );
		}
		if ( $model === '' ) {
			return array( 'ok' => false, 'error' => 'llm_model 不能为空' );
		}
		if ( ! is_array( $messages ) || empty( $messages ) ) {
			return array( 'ok' => false, 'error' => 'messages 不能为空' );
		}

		$endpoint = $base_url . '/chat/completions';

		$body = array(
			'model' => $model,
			'messages' => $messages,
			'temperature' => 0.2,
		);

		$max_tokens = (int) $max_tokens;
		if ( $max_tokens > 0 ) {
			// Keep a sane ceiling; different providers have different limits.
			$body['max_tokens'] = max( 1, min( 65536, $max_tokens ) );
		}

		$args = array(
			'timeout' => (int) $timeout,
			'headers' => array(
				'Content-Type'  => 'application/json; charset=utf-8',
				'Authorization' => 'Bearer ' . $api_key,
			),
			'body' => wp_json_encode( $body ),
		);

		$response = wp_remote_post( $endpoint, $args );
		if ( is_wp_error( $response ) ) {
			return array( 'ok' => false, 'error' => $response->get_error_message(), 'raw' => $response );
		}

		$code = (int) wp_remote_retrieve_response_code( $response );
		$raw_body = wp_remote_retrieve_body( $response );
		$data = json_decode( $raw_body, true );

		if ( $code < 200 || $code >= 300 ) {
			$error = 'LLM 请求失败（HTTP ' . $code . '）';
			if ( is_array( $data ) && isset( $data['error']['message'] ) ) {
				$error .= ': ' . (string) $data['error']['message'];
			}
			return array( 'ok' => false, 'error' => $error, 'raw' => $raw_body );
		}

		if ( ! is_array( $data ) ) {
			return array( 'ok' => false, 'error' => 'LLM 返回不是有效 JSON', 'raw' => $raw_body );
		}

		return array( 'ok' => true, 'data' => $data );
	}
}
