<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class WP_AIW_Image_Generator {
	private static $instance = null;

	/** @var WP_AIW_LLM_Client */
	private $llm;

	/** @var WP_AIW_Image_Client */
	private $image;

	/** @var WP_AIW_Image_Client_Gemini */
	private $image_gemini;

	/** @var WP_AIW_Image_Client_Zhipu */
	private $image_zhipu;

	public static function instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	private function __construct() {
		$this->llm = new WP_AIW_LLM_Client();
		$this->image = new WP_AIW_Image_Client();
		$this->image_gemini = new WP_AIW_Image_Client_Gemini();
		$this->image_zhipu = new WP_AIW_Image_Client_Zhipu();
	}

	/**
	 * @param array{title:string,requirements:string,language:string,style_hint?:string,size?:string,excerpt_text?:string} $input
	 * @return array{ok:bool, b64?:string, url?:string, mime?:string, prompt_used?:string, error?:string, debug?:array}
	 */
	public function generate_base64( $input ) {
		$settings = WP_AIW_Settings::instance()->get();

		if ( empty( $settings['image_enabled'] ) ) {
			return array( 'ok' => false, 'error' => '文生图功能未启用（image_enabled=0）' );
		}

		$title = isset( $input['title'] ) ? trim( (string) $input['title'] ) : '';
		if ( $title === '' ) {
			return array( 'ok' => false, 'error' => '标题不能为空' );
		}

		$language = isset( $input['language'] ) ? strtolower( trim( (string) $input['language'] ) ) : 'zh';
		if ( ! in_array( $language, array( 'zh', 'en' ), true ) ) {
			$language = 'zh';
		}

		$requirements = isset( $input['requirements'] ) ? trim( (string) $input['requirements'] ) : '';
		$style_hint = isset( $input['style_hint'] ) ? trim( (string) $input['style_hint'] ) : '';
		$excerpt_text = isset( $input['excerpt_text'] ) ? trim( (string) $input['excerpt_text'] ) : '';
		if ( $excerpt_text !== '' ) {
			if ( function_exists( 'mb_substr' ) ) {
				$excerpt_text = mb_substr( $excerpt_text, 0, 4000 );
			} else {
				$excerpt_text = substr( $excerpt_text, 0, 4000 );
			}
		}
		$size = isset( $input['size'] ) ? trim( (string) $input['size'] ) : '';
		if ( $size === '' ) {
			$size = isset( $settings['image_size'] ) ? (string) $settings['image_size'] : '1024x1024';
		}

		$prompt = $this->build_image_prompt_via_llm( $title, $requirements, $excerpt_text, $style_hint, $language, $settings );
		if ( empty( $prompt['ok'] ) ) {
			return array( 'ok' => false, 'error' => isset( $prompt['error'] ) ? (string) $prompt['error'] : '生成图片 prompt 失败', 'debug' => isset( $prompt['debug'] ) ? $prompt['debug'] : null );
		}

		$prompt_used = (string) $prompt['prompt'];

		$provider = isset( $settings['image_provider'] ) ? (string) $settings['image_provider'] : 'zhipu_glm_image';
		$b64 = '';
		$url = '';
		$mime = 'image/png';
		$debug_raw = null;

		if ( $provider === 'zhipu_glm_image' ) {
			$resp = $this->image_zhipu->generate_image_url(
				$settings['image_base_url'],
				$settings['image_api_key'],
				$settings['image_model'],
				$prompt_used,
				$size,
				isset( $settings['image_timeout_seconds'] ) ? (int) $settings['image_timeout_seconds'] : 180
			);
			if ( empty( $resp['ok'] ) ) {
				return array( 'ok' => false, 'error' => isset( $resp['error'] ) ? (string) $resp['error'] : '智谱调用失败', 'debug' => array( 'provider' => $provider, 'prompt_used' => $prompt_used, 'raw' => isset( $resp['raw'] ) ? $resp['raw'] : null ) );
			}
			$url = isset( $resp['url'] ) ? (string) $resp['url'] : '';
			$debug_raw = isset( $resp['raw'] ) ? $resp['raw'] : null;
			$mime = 'image/png';
		} elseif ( $provider === 'openai_compatible_images' ) {
			$resp = $this->image->generate_images(
				$settings['image_base_url'],
				$settings['image_api_key'],
				$settings['image_model'],
				$prompt_used,
				$size,
				isset( $settings['image_timeout_seconds'] ) ? (int) $settings['image_timeout_seconds'] : 180,
				isset( $settings['image_response_format'] ) ? (string) $settings['image_response_format'] : 'b64_json',
				isset( $settings['image_extra_params_json'] ) ? (string) $settings['image_extra_params_json'] : ''
			);

			if ( empty( $resp['ok'] ) ) {
				return array( 'ok' => false, 'error' => isset( $resp['error'] ) ? (string) $resp['error'] : 'Image 调用失败', 'debug' => array( 'provider' => $provider, 'prompt_used' => $prompt_used ) );
			}

			$data = $resp['data'];
			if ( isset( $data['data'][0]['b64_json'] ) ) {
				$b64 = (string) $data['data'][0]['b64_json'];
			} elseif ( isset( $data['data'][0]['url'] ) ) {
				// Some providers ignore response_format=b64_json and return an URL.
				$url = (string) $data['data'][0]['url'];
			}
			$debug_raw = $data;
			$mime = 'image/png';
		} else {
			// Default: Gemini generateContent
			$resp = $this->image_gemini->generate_image_base64(
				$settings['image_base_url'],
				$settings['image_api_key'],
				$settings['image_model'],
				$prompt_used,
				isset( $settings['image_timeout_seconds'] ) ? (int) $settings['image_timeout_seconds'] : 180
			);
			if ( empty( $resp['ok'] ) ) {
				return array( 'ok' => false, 'error' => isset( $resp['error'] ) ? (string) $resp['error'] : 'Gemini 调用失败', 'debug' => array( 'provider' => $provider, 'prompt_used' => $prompt_used, 'raw' => isset( $resp['raw'] ) ? $resp['raw'] : null ) );
			}
			$b64 = isset( $resp['b64'] ) ? (string) $resp['b64'] : '';
			$mime = isset( $resp['mime'] ) ? (string) $resp['mime'] : 'image/png';
			$debug_raw = isset( $resp['raw'] ) ? $resp['raw'] : null;
		}

		$b64 = trim( $b64 );
		$url = trim( $url );
		if ( $b64 === '' && $url === '' ) {
			return array( 'ok' => false, 'error' => 'Image 返回缺少图片数据', 'debug' => array( 'provider' => $provider, 'prompt_used' => $prompt_used, 'raw' => $debug_raw ) );
		}

		return array(
			'ok' => true,
			'b64' => $b64,
			'url' => $url,
			'mime' => $mime,
			'prompt_used' => $prompt_used,
			'debug' => array(
				'prompt_generation' => isset( $prompt['debug'] ) ? $prompt['debug'] : null,
				'provider' => $provider,
			),
		);
	}

	/**
	 * @return array{ok:bool, prompt?:string, error?:string, debug?:array}
	 */
	private function build_image_prompt_via_llm( $title, $requirements, $excerpt_text, $style_hint, $language, $settings ) {
		$title = trim( (string) $title );
		$requirements = trim( (string) $requirements );
		$excerpt_text = trim( (string) $excerpt_text );
		$style_hint = trim( (string) $style_hint );
		$language = ( $language === 'en' ) ? 'en' : 'zh';

		$prefix = ( $language === 'en' ) ? (string) $settings['image_prompt_prefix_en'] : (string) $settings['image_prompt_prefix_zh'];
		$prefix = trim( $prefix );
		if ( $prefix === '' ) {
			$prefix = ( $language === 'en' )
				? 'Generate a concise English prompt for a technical illustration. Avoid text.'
				: '生成适合技术文章插图的英文 prompt，避免文字/水印。';
		}

		$system = $prefix;
		$rules = ( $language === 'en' )
			? "Output ONLY one JSON object. Field: prompt (string, required). The prompt MUST be in English."
			: "只能输出 1 个 JSON 对象。字段：prompt（string，必填）。prompt 必须是英文。";

		$user = ( $language === 'en' )
			? "Title:\n" . $title . "\n\nRequirements:\n" . ( $requirements !== '' ? $requirements : '(none)' )
			: "标题：\n" . $title . "\n\n写作要求：\n" . ( $requirements !== '' ? $requirements : '（无）' );

		if ( $excerpt_text !== '' ) {
			$user .= ( $language === 'en' )
				? "\n\nExcerpt (important, based on the article content):\n" . $excerpt_text
				: "\n\n内容摘抄（重要，来自文章正文）：\n" . $excerpt_text;
		}

		if ( $style_hint !== '' ) {
			$user .= ( $language === 'en' ) ? "\n\nStyle hint:\n" . $style_hint : "\n\n风格提示：\n" . $style_hint;
		}

		$user .= "\n\n" . $rules;

		$messages = array(
			array( 'role' => 'system', 'content' => $system ),
			array( 'role' => 'user', 'content' => $user ),
		);

		$resp = $this->llm->chat_completions(
			$settings['llm_base_url'],
			$settings['llm_api_key'],
			$settings['llm_model'],
			$messages,
			isset( $settings['llm_timeout_seconds'] ) ? (int) $settings['llm_timeout_seconds'] : 60,
			isset( $settings['llm_max_tokens'] ) ? (int) $settings['llm_max_tokens'] : 0
		);

		if ( empty( $resp['ok'] ) ) {
			return array( 'ok' => false, 'error' => isset( $resp['error'] ) ? (string) $resp['error'] : 'LLM 调用失败' );
		}

		$text = '';
		$data = $resp['data'];
		if ( isset( $data['choices'][0]['message']['content'] ) ) {
			$text = (string) $data['choices'][0]['message']['content'];
		}
		$text = trim( $text );
		if ( $text === '' ) {
			return array( 'ok' => false, 'error' => 'LLM 返回为空' );
		}

		$json_text = $this->extract_first_json_object( $text );
		$parsed = $json_text !== '' ? json_decode( $json_text, true ) : null;
		if ( ! is_array( $parsed ) ) {
			$err = function_exists( 'json_last_error_msg' ) ? json_last_error_msg() : 'JSON parse error';
			return array( 'ok' => false, 'error' => 'prompt JSON 解析失败：' . $err, 'debug' => array( 'raw' => $text ) );
		}

		$prompt = isset( $parsed['prompt'] ) ? trim( (string) $parsed['prompt'] ) : '';
		if ( $prompt === '' ) {
			return array( 'ok' => false, 'error' => 'prompt JSON 缺少 prompt', 'debug' => array( 'raw' => $text ) );
		}

		return array( 'ok' => true, 'prompt' => $prompt, 'debug' => array( 'raw' => $text ) );
	}

	/**
	 * Extract the first JSON object from LLM output.
	 */
	private function extract_first_json_object( $text ) {
		$text = is_string( $text ) ? trim( $text ) : '';
		if ( $text === '' ) {
			return '';
		}

		if ( preg_match( '/```(?:json)?\s*([\s\S]*?)\s*```/i', $text, $m ) ) {
			$candidate = trim( (string) $m[1] );
			if ( $candidate !== '' ) {
				$text = $candidate;
			}
		}

		$start = strpos( $text, '{' );
		if ( $start === false ) {
			return '';
		}

		$depth = 0;
		$in_string = false;
		$escape = false;
		$len = strlen( $text );
		for ( $i = $start; $i < $len; $i++ ) {
			$ch = $text[ $i ];
			if ( $in_string ) {
				if ( $escape ) {
					$escape = false;
					continue;
				}
				if ( $ch === '\\' ) {
					$escape = true;
					continue;
				}
				if ( $ch === '"' ) {
					$in_string = false;
				}
				continue;
			}

			if ( $ch === '"' ) {
				$in_string = true;
				continue;
			}

			if ( $ch === '{' ) {
				$depth++;
				continue;
			}
			if ( $ch === '}' ) {
				$depth--;
				if ( $depth === 0 ) {
					return substr( $text, $start, $i - $start + 1 );
				}
			}
		}

		return '';
	}
}
