<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class WP_AIW_Image_Client_OpenAI_Compatible {
	/**
	 * OpenAI-compatible images/generations call (configurable response_format & extra params).
	 *
	 * @return array{ok:bool, data?:array, error?:string, raw?:mixed}
	 */
	public function generate_images( $base_url, $api_key, $model, $prompt, $size = '1024x1024', $timeout = 180, $response_format = 'b64_json', $extra_params_json = '' ) {
		$base_url = is_string( $base_url ) ? rtrim( trim( $base_url ), '/' ) : '';
		$api_key  = is_string( $api_key ) ? trim( $api_key ) : '';
		$model    = is_string( $model ) ? trim( $model ) : '';
		$prompt   = is_string( $prompt ) ? trim( $prompt ) : '';
		$size     = is_string( $size ) ? trim( $size ) : '1024x1024';
		$response_format = is_string( $response_format ) ? trim( $response_format ) : 'b64_json';
		$extra_params_json = is_string( $extra_params_json ) ? trim( $extra_params_json ) : '';

		if ( $base_url === '' ) {
			return array( 'ok' => false, 'error' => 'image_base_url 未配置' );
		}
		if ( $api_key === '' ) {
			return array( 'ok' => false, 'error' => 'image_api_key 未配置' );
		}
		if ( $model === '' ) {
			return array( 'ok' => false, 'error' => 'image_model 未配置' );
		}
		if ( $prompt === '' ) {
			return array( 'ok' => false, 'error' => 'prompt 不能为空' );
		}

		$allowed_format = array( 'b64_json', 'url' );
		if ( ! in_array( $response_format, $allowed_format, true ) ) {
			$response_format = 'b64_json';
		}

		$endpoint = $base_url . '/images/generations';

		$body = array(
			'model' => $model,
			'prompt' => $prompt,
			'size' => $size,
			'n' => 1,
			'response_format' => $response_format,
		);

		if ( $extra_params_json !== '' ) {
			$extra = json_decode( $extra_params_json, true );
			if ( ! is_array( $extra ) ) {
				$err = function_exists( 'json_last_error_msg' ) ? json_last_error_msg() : 'JSON parse error';
				return array( 'ok' => false, 'error' => 'image_extra_params_json JSON 解析失败：' . $err );
			}

			// Require JSON object (associative array).
			$keys = array_keys( $extra );
			$is_assoc = array_keys( $keys ) !== $keys;
			if ( ! $is_assoc ) {
				return array( 'ok' => false, 'error' => 'image_extra_params_json 必须是 JSON 对象（{}），不能是数组（[]）' );
			}

			$body = array_merge( $body, $extra );
		}

		$args = array(
			'timeout' => (int) $timeout,
			'redirection' => 3,
			'reject_unsafe_urls' => true,
			'headers' => array(
				'Content-Type'  => 'application/json; charset=utf-8',
				'Authorization' => 'Bearer ' . $api_key,
			),
			'body' => wp_json_encode( $body ),
		);

		$response = wp_remote_post( $endpoint, $args );
		if ( is_wp_error( $response ) ) {
			return array( 'ok' => false, 'error' => $response->get_error_message(), 'raw' => $response );
		}

		$code = (int) wp_remote_retrieve_response_code( $response );
		$raw_body = wp_remote_retrieve_body( $response );
		$data = json_decode( $raw_body, true );

		if ( $code < 200 || $code >= 300 ) {
			$error = 'Image 请求失败（HTTP ' . $code . '）';
			if ( is_array( $data ) && isset( $data['error']['message'] ) ) {
				$error .= ': ' . (string) $data['error']['message'];
			}
			return array( 'ok' => false, 'error' => $error, 'raw' => $raw_body );
		}

		if ( ! is_array( $data ) ) {
			return array( 'ok' => false, 'error' => 'Image 返回不是有效 JSON', 'raw' => $raw_body );
		}

		return array( 'ok' => true, 'data' => $data );
	}
}
