<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class WP_AIW_Image_Client_Gemini {
	/**
	 * Gemini generateContent call (supports image models).
	 *
	 * @return array{ok:bool, b64?:string, mime?:string, error?:string, raw?:mixed}
	 */
	public function generate_image_base64( $base_url, $api_key, $model, $prompt, $timeout = 180 ) {
		$base_url = is_string( $base_url ) ? rtrim( trim( $base_url ), '/' ) : '';
		$api_key = is_string( $api_key ) ? trim( $api_key ) : '';
		$model = is_string( $model ) ? trim( $model ) : '';
		$prompt = is_string( $prompt ) ? trim( $prompt ) : '';

		if ( $base_url === '' ) {
			$base_url = 'https://generativelanguage.googleapis.com';
		}
		if ( $api_key === '' ) {
			return array( 'ok' => false, 'error' => 'image_api_key 未配置' );
		}
		if ( $model === '' ) {
			return array( 'ok' => false, 'error' => 'image_model 未配置（Gemini: 例如 gemini-2.5-flash-image）' );
		}
		if ( $prompt === '' ) {
			return array( 'ok' => false, 'error' => 'prompt 不能为空' );
		}

		// Accept either "gemini-2.5-flash-image" or "models/gemini-2.5-flash-image".
		if ( strpos( $model, 'models/' ) === 0 ) {
			$model = substr( $model, strlen( 'models/' ) );
		}

		$endpoint = $base_url . '/v1beta/models/' . rawurlencode( $model ) . ':generateContent';

		$body = array(
			'contents' => array(
				array(
					'parts' => array(
						array( 'text' => $prompt ),
					),
				),
			),
		);

		$args = array(
			'timeout' => (int) $timeout,
			'redirection' => 3,
			'reject_unsafe_urls' => true,
			'headers' => array(
				'Content-Type' => 'application/json; charset=utf-8',
				'x-goog-api-key' => $api_key,
			),
			'body' => wp_json_encode( $body ),
		);

		$response = wp_remote_post( $endpoint, $args );
		if ( is_wp_error( $response ) ) {
			return array( 'ok' => false, 'error' => $response->get_error_message(), 'raw' => $response );
		}

		$code = (int) wp_remote_retrieve_response_code( $response );
		$raw_body = wp_remote_retrieve_body( $response );
		$data = json_decode( $raw_body, true );

		if ( $code < 200 || $code >= 300 ) {
			$error = 'Gemini 请求失败（HTTP ' . $code . '）';
			if ( is_array( $data ) && isset( $data['error']['message'] ) ) {
				$error .= ': ' . (string) $data['error']['message'];
			}
			return array( 'ok' => false, 'error' => $error, 'raw' => $raw_body );
		}

		if ( ! is_array( $data ) ) {
			return array( 'ok' => false, 'error' => 'Gemini 返回不是有效 JSON', 'raw' => $raw_body );
		}

		// Try to locate inline image data.
		$b64 = '';
		$mime = '';

		if ( isset( $data['candidates'][0]['content']['parts'] ) && is_array( $data['candidates'][0]['content']['parts'] ) ) {
			foreach ( $data['candidates'][0]['content']['parts'] as $part ) {
				if ( ! is_array( $part ) ) {
					continue;
				}

				// Newer naming: inlineData { mimeType, data }
				if ( isset( $part['inlineData'] ) && is_array( $part['inlineData'] ) ) {
					$mime = isset( $part['inlineData']['mimeType'] ) ? (string) $part['inlineData']['mimeType'] : '';
					$b64 = isset( $part['inlineData']['data'] ) ? (string) $part['inlineData']['data'] : '';
					break;
				}

				// Older naming: inline_data { mime_type, data }
				if ( isset( $part['inline_data'] ) && is_array( $part['inline_data'] ) ) {
					$mime = isset( $part['inline_data']['mime_type'] ) ? (string) $part['inline_data']['mime_type'] : '';
					$b64 = isset( $part['inline_data']['data'] ) ? (string) $part['inline_data']['data'] : '';
					break;
				}
			}
		}

		$b64 = trim( $b64 );
		$mime = trim( $mime );
		if ( $b64 === '' ) {
			return array( 'ok' => false, 'error' => 'Gemini 返回缺少图片数据（inlineData/inline_data）', 'raw' => $data );
		}

		if ( $mime === '' ) {
			$mime = 'image/png';
		}

		return array(
			'ok' => true,
			'b64' => $b64,
			'mime' => $mime,
			'raw' => $data,
		);
	}
}
