<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class WP_AIW_Image_Admin {
	private static $instance = null;

	public static function instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	private function __construct() {
		add_action( 'wp_ajax_wp_aiw_generate_image', array( $this, 'ajax_generate_image' ) );
		add_action( 'wp_ajax_wp_aiw_upload_image', array( $this, 'ajax_upload_image' ) );
		add_action( 'wp_ajax_wp_aiw_set_featured_image', array( $this, 'ajax_set_featured_image' ) );
	}

	/**
	 * Delete current featured image and its attachment (force delete).
	 *
	 * @param int $post_id
	 * @return array{ok:bool, attachment_id?:int, error?:string}
	 */
	public function delete_featured_image_and_attachment( $post_id ) {
		$post_id = (int) $post_id;
		if ( $post_id <= 0 ) {
			return array( 'ok' => false, 'error' => 'post_id 无效' );
		}

		$thumb_id = (int) get_post_thumbnail_id( $post_id );
		if ( $thumb_id <= 0 ) {
			return array( 'ok' => true, 'attachment_id' => 0 );
		}

		if ( function_exists( 'delete_post_thumbnail' ) ) {
			delete_post_thumbnail( $post_id );
		} else {
			delete_post_meta( $post_id, '_thumbnail_id' );
		}

		$deleted = wp_delete_attachment( $thumb_id, true );
		if ( ! $deleted ) {
			return array( 'ok' => false, 'attachment_id' => $thumb_id, 'error' => '删除附件失败（可能已被删除或无权限）' );
		}

		return array( 'ok' => true, 'attachment_id' => $thumb_id );
	}

	/**
	 * Generate image via provider -> upload to media library -> set as featured image.
	 *
	 * @param array{post_id:int,title:string,requirements:string,language:string,excerpt_text?:string,style_hint?:string,size?:string} $opts
	 * @return array{ok:bool, attachment_id?:int, error?:string}
	 */
	public function generate_upload_and_set_featured_image( $opts ) {
		$opts = is_array( $opts ) ? $opts : array();
		$post_id = isset( $opts['post_id'] ) ? (int) $opts['post_id'] : 0;
		$title = isset( $opts['title'] ) ? (string) $opts['title'] : '';
		$requirements = isset( $opts['requirements'] ) ? (string) $opts['requirements'] : '';
		$language = isset( $opts['language'] ) ? (string) $opts['language'] : 'zh';
		$excerpt_text = isset( $opts['excerpt_text'] ) ? (string) $opts['excerpt_text'] : '';
		$style_hint = isset( $opts['style_hint'] ) ? (string) $opts['style_hint'] : '';
		$size = isset( $opts['size'] ) ? (string) $opts['size'] : '';
		if ( $language !== 'en' ) {
			$language = 'zh';
		}

		if ( $post_id <= 0 ) {
			return array( 'ok' => false, 'error' => 'post_id 无效' );
		}
		$title = trim( $title );
		if ( $title === '' ) {
			return array( 'ok' => false, 'error' => '标题不能为空' );
		}

		$generator = WP_AIW_Image_Generator::instance();
		$r = $generator->generate_base64(
			array(
				'title' => $title,
				'requirements' => trim( $requirements ),
				'language' => $language,
				'excerpt_text' => trim( $excerpt_text ),
				'style_hint' => trim( $style_hint ),
				'size' => trim( $size ),
			)
		);
		if ( empty( $r['ok'] ) ) {
			return array( 'ok' => false, 'error' => isset( $r['error'] ) ? (string) $r['error'] : '生成失败' );
		}

		$mime = isset( $r['mime'] ) ? (string) $r['mime'] : 'image/png';
		$b64 = isset( $r['b64'] ) ? (string) $r['b64'] : '';
		$url = isset( $r['url'] ) ? (string) $r['url'] : '';

		$up = $this->upload_to_media_library( $post_id, $b64, $url, $mime, $title, $title );
		if ( empty( $up['ok'] ) ) {
			return array( 'ok' => false, 'error' => isset( $up['error'] ) ? (string) $up['error'] : '入库失败' );
		}

		$attach_id = isset( $up['attachment_id'] ) ? (int) $up['attachment_id'] : 0;
		if ( $attach_id <= 0 ) {
			return array( 'ok' => false, 'error' => '入库成功但缺少 attachment_id' );
		}

		$ok = set_post_thumbnail( $post_id, $attach_id );
		if ( ! $ok ) {
			return array( 'ok' => false, 'error' => '设置特色图失败' );
		}

		return array( 'ok' => true, 'attachment_id' => $attach_id );
	}

	public function ajax_set_featured_image() {
		if ( ! current_user_can( 'edit_posts' ) && ! current_user_can( 'edit_pages' ) ) {
			wp_send_json_error( array( 'message' => '无权限' ), 403 );
		}
		check_ajax_referer( 'wp_aiw_nonce', 'nonce' );

		$post_id = isset( $_POST['postId'] ) ? (int) $_POST['postId'] : 0;
		$attachment_id = isset( $_POST['attachmentId'] ) ? (int) $_POST['attachmentId'] : 0;

		if ( $post_id <= 0 ) {
			wp_send_json_error( array( 'message' => 'postId 无效（请先保存文章/草稿）' ), 400 );
		}
		if ( $attachment_id <= 0 ) {
			wp_send_json_error( array( 'message' => 'attachmentId 无效' ), 400 );
		}
		if ( ! current_user_can( 'edit_post', $post_id ) ) {
			wp_send_json_error( array( 'message' => '无权限编辑该文章' ), 403 );
		}

		$ok = set_post_thumbnail( $post_id, $attachment_id );
		if ( ! $ok ) {
			wp_send_json_error( array( 'message' => '设置特色图失败' ), 500 );
		}

		// Return refreshed HTML so UI can update immediately.
		require_once ABSPATH . 'wp-admin/includes/post.php';
		$html = _wp_post_thumbnail_html( $attachment_id, $post_id );

		wp_send_json_success(
			array(
				'postId' => (int) $post_id,
				'attachmentId' => (int) $attachment_id,
				'html' => (string) $html,
			)
		);
	}

	public function ajax_generate_image() {
		if ( ! current_user_can( 'edit_posts' ) && ! current_user_can( 'edit_pages' ) ) {
			wp_send_json_error( array( 'message' => '无权限' ), 403 );
		}
		check_ajax_referer( 'wp_aiw_nonce', 'nonce' );

		$title = isset( $_POST['title'] ) ? sanitize_text_field( wp_unslash( $_POST['title'] ) ) : '';
		$requirements = isset( $_POST['requirements'] ) ? wp_unslash( (string) $_POST['requirements'] ) : '';
		$requirements = is_string( $requirements ) ? trim( $requirements ) : '';
		$language = isset( $_POST['language'] ) ? sanitize_key( wp_unslash( $_POST['language'] ) ) : 'zh';
		if ( ! in_array( $language, array( 'zh', 'en' ), true ) ) {
			$language = 'zh';
		}

		$style_hint = isset( $_POST['styleHint'] ) ? wp_unslash( (string) $_POST['styleHint'] ) : '';
		$style_hint = is_string( $style_hint ) ? trim( $style_hint ) : '';
		$excerpt_text = isset( $_POST['excerptText'] ) ? wp_unslash( (string) $_POST['excerptText'] ) : '';
		$excerpt_text = is_string( $excerpt_text ) ? trim( $excerpt_text ) : '';
		if ( $excerpt_text !== '' && function_exists( 'mb_substr' ) ) {
			$excerpt_text = mb_substr( $excerpt_text, 0, 4000 );
		} elseif ( $excerpt_text !== '' ) {
			$excerpt_text = substr( $excerpt_text, 0, 4000 );
		}
		$size = isset( $_POST['size'] ) ? sanitize_text_field( wp_unslash( $_POST['size'] ) ) : '';

		$generator = WP_AIW_Image_Generator::instance();
		$r = $generator->generate_base64(
			array(
				'title' => $title,
				'requirements' => $requirements,
				'language' => $language,
				'excerpt_text' => $excerpt_text,
				'style_hint' => $style_hint,
				'size' => $size,
			)
		);

		if ( empty( $r['ok'] ) ) {
			wp_send_json_error( array( 'message' => isset( $r['error'] ) ? (string) $r['error'] : '生成失败', 'debug' => isset( $r['debug'] ) ? $r['debug'] : null ), 500 );
		}

		$payload = array(
			'mime' => isset( $r['mime'] ) ? (string) $r['mime'] : 'image/png',
			'b64' => isset( $r['b64'] ) ? (string) $r['b64'] : '',
			'url' => isset( $r['url'] ) ? (string) $r['url'] : '',
			'promptUsed' => isset( $r['prompt_used'] ) ? (string) $r['prompt_used'] : '',
		);

		$want_debug = ! empty( $_POST['debug'] );
		if ( $want_debug && isset( $r['debug'] ) ) {
			$payload['debug'] = $r['debug'];
		}

		wp_send_json_success( $payload );
	}

	public function ajax_upload_image() {
		if ( ! current_user_can( 'edit_posts' ) && ! current_user_can( 'edit_pages' ) ) {
			wp_send_json_error( array( 'message' => '无权限' ), 403 );
		}
		check_ajax_referer( 'wp_aiw_nonce', 'nonce' );

		$settings = WP_AIW_Settings::instance()->get();
		if ( empty( $settings['image_enabled'] ) ) {
			wp_send_json_error( array( 'message' => '图片功能未启用（image_enabled=0）' ), 400 );
		}

		$post_id = isset( $_POST['postId'] ) ? (int) $_POST['postId'] : 0;
		$b64 = isset( $_POST['b64'] ) ? (string) wp_unslash( $_POST['b64'] ) : '';
		$url = isset( $_POST['url'] ) ? (string) wp_unslash( $_POST['url'] ) : '';
		$mime = isset( $_POST['mime'] ) ? sanitize_text_field( wp_unslash( $_POST['mime'] ) ) : 'image/png';
		$title = isset( $_POST['title'] ) ? sanitize_text_field( wp_unslash( $_POST['title'] ) ) : '';
		$alt = isset( $_POST['alt'] ) ? sanitize_text_field( wp_unslash( $_POST['alt'] ) ) : '';

		$b64 = trim( $b64 );
		$url = is_string( $url ) ? trim( $url ) : '';
		if ( $b64 === '' && $url === '' ) {
			wp_send_json_error( array( 'message' => '缺少 b64 或 url' ), 400 );
		}

		// Basic allowlist
		$allowed_mime = array( 'image/png', 'image/jpeg', 'image/webp' );
		if ( ! in_array( $mime, $allowed_mime, true ) ) {
			wp_send_json_error( array( 'message' => '不支持的 mime：' . $mime ), 400 );
		}

		$up = $this->upload_to_media_library( $post_id, $b64, $url, $mime, $title, $alt );
		if ( empty( $up['ok'] ) ) {
			wp_send_json_error( array( 'message' => isset( $up['error'] ) ? (string) $up['error'] : '入库失败' ), 500 );
		}

		wp_send_json_success(
			array(
				'attachmentId' => (int) $up['attachment_id'],
				'url' => (string) $up['url'],
				'mime' => (string) $up['mime'],
			)
		);
	}

	/**
	 * @return void
	 */
	private function ensure_media_includes() {
		require_once ABSPATH . 'wp-admin/includes/image.php';
		require_once ABSPATH . 'wp-admin/includes/file.php';
		require_once ABSPATH . 'wp-admin/includes/media.php';
	}

	/**
	 * Upload by base64 or url into media library.
	 *
	 * @param int $post_id
	 * @param string $b64
	 * @param string $url
	 * @param string $mime
	 * @param string $title
	 * @param string $alt
	 * @return array{ok:bool, attachment_id?:int, url?:string, mime?:string, error?:string}
	 */
	private function upload_to_media_library( $post_id, $b64, $url, $mime, $title, $alt ) {
		$post_id = (int) $post_id;
		$b64 = is_string( $b64 ) ? trim( $b64 ) : '';
		$url = is_string( $url ) ? trim( $url ) : '';
		$mime = is_string( $mime ) ? trim( $mime ) : 'image/png';
		$title = is_string( $title ) ? trim( $title ) : '';
		$alt = is_string( $alt ) ? trim( $alt ) : '';

		if ( $b64 === '' && $url === '' ) {
			return array( 'ok' => false, 'error' => '缺少 b64 或 url' );
		}

		$allowed_mime = array( 'image/png', 'image/jpeg', 'image/webp' );
		if ( ! in_array( $mime, $allowed_mime, true ) ) {
			return array( 'ok' => false, 'error' => '不支持的 mime：' . $mime );
		}

		$this->ensure_media_includes();

		$attach_id = 0;
		$final_mime = $mime;
		$final_url = '';

		if ( $b64 !== '' ) {
			$raw = base64_decode( $b64, true );
			if ( $raw === false ) {
				return array( 'ok' => false, 'error' => 'b64 解码失败' );
			}

			if ( strlen( $raw ) > 8 * 1024 * 1024 ) {
				return array( 'ok' => false, 'error' => '图片过大（> 8MB）' );
			}

			$ext = 'png';
			if ( $mime === 'image/jpeg' ) {
				$ext = 'jpg';
			} elseif ( $mime === 'image/webp' ) {
				$ext = 'webp';
			}
			$filename = 'aiw-image-' . gmdate( 'Ymd-His' ) . '-' . wp_generate_password( 6, false, false ) . '.' . $ext;

			$upload = wp_upload_bits( $filename, null, $raw );
			if ( ! empty( $upload['error'] ) ) {
				return array( 'ok' => false, 'error' => '写入上传目录失败：' . (string) $upload['error'] );
			}

			$attachment = array(
				'post_mime_type' => $mime,
				'post_title' => $title !== '' ? $title : $filename,
				'post_content' => '',
				'post_status' => 'inherit',
			);

			$attach_id = wp_insert_attachment( $attachment, $upload['file'], $post_id > 0 ? $post_id : 0 );
			if ( is_wp_error( $attach_id ) ) {
				return array( 'ok' => false, 'error' => '入库失败：' . $attach_id->get_error_message() );
			}

			$meta = wp_generate_attachment_metadata( $attach_id, $upload['file'] );
			if ( is_array( $meta ) ) {
				wp_update_attachment_metadata( $attach_id, $meta );
			}
			$final_url = isset( $upload['url'] ) ? (string) $upload['url'] : '';
		} else {
			$url = $this->normalize_and_validate_url( $url );
			if ( $url === '' ) {
				return array( 'ok' => false, 'error' => 'URL 不合法或不允许下载' );
			}

			$tmp = download_url( $url, 60 );
			if ( is_wp_error( $tmp ) ) {
				return array( 'ok' => false, 'error' => '下载失败：' . $tmp->get_error_message() );
			}

			$sz = @filesize( $tmp );
			if ( is_int( $sz ) && $sz > 8 * 1024 * 1024 ) {
				@unlink( $tmp );
				return array( 'ok' => false, 'error' => '图片过大（> 8MB）' );
			}

			$path = wp_parse_url( $url, PHP_URL_PATH );
			$name = is_string( $path ) && $path !== '' ? basename( $path ) : '';
			if ( $name === '' ) {
				$name = 'aiw-image-' . gmdate( 'Ymd-His' ) . '.png';
			}

			$file_array = array(
				'name' => $name,
				'tmp_name' => $tmp,
			);

			$attach_id = media_handle_sideload( $file_array, $post_id > 0 ? $post_id : 0, $title !== '' ? $title : null );
			if ( is_wp_error( $attach_id ) ) {
				@unlink( $tmp );
				return array( 'ok' => false, 'error' => '入库失败：' . $attach_id->get_error_message() );
			}

			$final_mime = (string) get_post_mime_type( $attach_id );
			$final_url = (string) wp_get_attachment_url( $attach_id );
		}

		if ( $attach_id <= 0 ) {
			return array( 'ok' => false, 'error' => '入库失败：attachment_id=0' );
		}

		if ( $alt !== '' ) {
			update_post_meta( $attach_id, '_wp_attachment_image_alt', $alt );
		}

		if ( $final_url === '' ) {
			$final_url = (string) wp_get_attachment_url( $attach_id );
		}

		return array(
			'ok' => true,
			'attachment_id' => (int) $attach_id,
			'url' => (string) $final_url,
			'mime' => (string) $final_mime,
		);
	}

	private function normalize_and_validate_url( $url ) {
		$url = is_string( $url ) ? trim( $url ) : '';
		if ( $url === '' ) {
			return '';
		}

		if ( function_exists( 'wp_http_validate_url' ) ) {
			$validated = wp_http_validate_url( $url );
			if ( ! $validated ) {
				return '';
			}
			$url = (string) $validated;
		}

		$parts = wp_parse_url( $url );
		if ( ! is_array( $parts ) ) {
			return '';
		}

		$scheme = isset( $parts['scheme'] ) ? strtolower( (string) $parts['scheme'] ) : '';
		if ( ! in_array( $scheme, array( 'http', 'https' ), true ) ) {
			return '';
		}

		$host = isset( $parts['host'] ) ? strtolower( (string) $parts['host'] ) : '';
		if ( $host === '' ) {
			return '';
		}

		if ( $host === 'localhost' || $host === '127.0.0.1' || $host === '0.0.0.0' ) {
			return '';
		}

		$ip = gethostbyname( $host );
		if ( is_string( $ip ) && $ip !== '' && $ip !== $host && filter_var( $ip, FILTER_VALIDATE_IP ) ) {
			if ( filter_var( $ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE ) === false ) {
				return '';
			}
		}

		return esc_url_raw( $url );
	}
}
