<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class WP_AIW_Admin {
	private static $instance = null;

	private function get_home_settings_summary() {
		$opt_name = 'wp_aiw_settings';
		try {
			if ( class_exists( 'WP_AIW_Settings' ) ) {
				$opt_name = WP_AIW_Settings::instance()->get_option_name();
			}
		} catch ( Exception $e ) {
			// ignore
		}

		$raw = get_option( $opt_name, array() );
		if ( ! is_array( $raw ) ) {
			$raw = array();
		}

		$out = array(
			'llm_model'     => 'deepseek-chat',
			'image_model'   => 'glm-image',
			'image_enabled' => 0,
		);

		if ( isset( $raw['llm_model'] ) ) {
			$out['llm_model'] = is_string( $raw['llm_model'] ) ? trim( (string) $raw['llm_model'] ) : $out['llm_model'];
		}
		if ( isset( $raw['image_model'] ) ) {
			$out['image_model'] = is_string( $raw['image_model'] ) ? trim( (string) $raw['image_model'] ) : $out['image_model'];
		}
		$out['image_enabled'] = ! empty( $raw['image_enabled'] ) ? 1 : 0;

		if ( $out['llm_model'] === '' ) {
			$out['llm_model'] = 'deepseek-chat';
		}
		if ( $out['image_model'] === '' ) {
			$out['image_model'] = 'glm-image';
		}

		return $out;
	}

	public static function instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	private function __construct() {
		add_action( 'admin_menu', array( $this, 'register_pages' ) );
		add_action( 'admin_menu', array( $this, 'reorder_submenus' ), 999 );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_assets' ) );

		add_action( 'wp_ajax_wp_aiw_process_next', array( $this, 'ajax_process_next' ) );
		add_action( 'wp_ajax_wp_aiw_featured_batch_next', array( $this, 'ajax_featured_batch_next' ) );
	}

	public function register_pages() {
		add_menu_page(
			'AI Writer',
			'AI Writer',
			'manage_options',
			'wp-aiw',
			array( $this, 'render_home_page' ),
			'dashicons-edit',
			58
		);

		add_submenu_page(
			'wp-aiw',
			'批量重写优化',
			'批量重写优化',
			'manage_options',
			'wp-aiw-runner',
			array( $this, 'render_runner_page' )
		);

		add_submenu_page(
			'wp-aiw',
			'批量AI特色图生成',
			'批量AI特色图生成',
			'manage_options',
			'wp-aiw-featured-batch',
			array( $this, 'render_featured_batch_page' )
		);

		add_submenu_page(
			'wp-aiw',
			'设置',
			'设置',
			'manage_options',
			'wp-aiw-settings',
			array( $this, 'render_settings_page' )
		);
	}

	public function render_home_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}
		$settings = $this->get_home_settings_summary();
		$items = array(
			array(
				'title' => 'AI 写文章',
				'desc' => '输入标题与要求，生成草稿；可选自动特色图。',
				'page' => 'wp-aiw-writer',
				'icon' => 'dashicons-welcome-write-blog',
			),
			array(
				'title' => 'URL 抓取文章',
				'desc' => '输入 URL 自动抓取清洗后生成草稿。',
				'page' => 'wp-aiw-writer-from-url',
				'icon' => 'dashicons-admin-links',
			),
			array(
				'title' => 'AI 批量写文章',
				'desc' => '每行一个任务，支持暂停/继续/刷新恢复队列。',
				'page' => 'wp-aiw-writer-batch',
				'icon' => 'dashicons-list-view',
			),
			array(
				'title' => '批量重写优化',
				'desc' => '按筛选范围并发重写正文与补充标签。',
				'page' => 'wp-aiw-runner',
				'icon' => 'dashicons-update',
			),
			array(
				'title' => '批量AI特色图生成',
				'desc' => '按筛选范围并发生成/重建文章特色图。',
				'page' => 'wp-aiw-featured-batch',
				'icon' => 'dashicons-format-image',
			),
			array(
				'title' => '设置',
				'desc' => '配置 LLM 与文生图 Provider、并发等参数。',
				'page' => 'wp-aiw-settings',
				'icon' => 'dashicons-admin-generic',
			),
		);
		?>
		<div class="wrap">
			<style>
				.wp-aiw-home-grid {
					display: grid;
					grid-template-columns: repeat( auto-fit, minmax( 260px, 1fr ) );
					gap: 14px;
					margin-top: 14px;
				}
				.wp-aiw-home-extras-grid {
					display: grid;
					grid-template-columns: repeat( 3, 1fr );
					gap: 14px;
					margin-top: 14px;
					align-items: stretch;
				}
				@media (max-width: 980px) {
					.wp-aiw-home-extras-grid {
						grid-template-columns: 1fr;
					}
				}
				.wp-aiw-extra-card {
					background: #fff;
					border: 1px solid #dcdcde;
					border-radius: 10px;
					padding: 14px;
					box-shadow: 0 1px 2px rgba(0,0,0,.04);
					display: flex;
					flex-direction: column;
					min-height: 180px;
				}
				.wp-aiw-extra-card h2 {
					margin: 0 0 10px;
					font-size: 14px;
					line-height: 1.4;
				}
				.wp-aiw-kv {
					display: grid;
					grid-template-columns: auto 1fr;
					gap: 6px 12px;
					align-items: start;
				}
				.wp-aiw-kv .k {
					color: #50575e;
					white-space: nowrap;
				}
				.wp-aiw-kv .v {
					color: #1d2327;
					word-break: break-word;
				}
				.wp-aiw-ann-list {
					margin: 0;
					padding-left: 18px;
				}
				.wp-aiw-ann-list li { margin: 6px 0; }
				.wp-aiw-home-card {
					display: block;
					text-decoration: none;
					background: #fff;
					border: 1px solid #dcdcde;
					border-radius: 10px;
					padding: 14px;
					box-shadow: 0 1px 2px rgba(0,0,0,.04);
					transition: transform .08s ease, box-shadow .08s ease, border-color .08s ease;
				}
				.wp-aiw-home-card:hover {
					transform: translateY(-1px);
					box-shadow: 0 6px 18px rgba(0,0,0,.06);
					border-color: #c3c4c7;
				}
				.wp-aiw-home-card h3 {
					margin: 0;
					font-size: 16px;
					line-height: 1.4;
				}
				.wp-aiw-home-card p {
					margin: 8px 0 0;
					color: #50575e;
				}
				.wp-aiw-home-card .wp-aiw-home-icon {
					width: 36px;
					height: 36px;
					display: inline-flex;
					align-items: center;
					justify-content: center;
					border-radius: 10px;
					background: #f6f7f7;
					margin-right: 10px;
					flex: 0 0 36px;
				}
				.wp-aiw-home-card .wp-aiw-home-row {
					display: flex;
					align-items: flex-start;
					gap: 10px;
				}
				.wp-aiw-home-card .dashicons {
					font-size: 18px;
					line-height: 18px;
					width: 18px;
					height: 18px;
					color: #1d2327;
				}
			</style>
			<h1>AI Writer</h1>
			<p class="description">从这里进入各个功能页面。默认所有操作都以文章标题为核心输入；批量任务支持暂停/继续与并发。</p>

			<div class="wp-aiw-home-grid">
				<?php foreach ( $items as $it ) : ?>
					<a class="wp-aiw-home-card" href="<?php echo esc_url( admin_url( 'admin.php?page=' . $it['page'] ) ); ?>">
						<div class="wp-aiw-home-row">
							<div class="wp-aiw-home-icon"><span class="dashicons <?php echo esc_attr( $it['icon'] ); ?>"></span></div>
							<div>
								<h3><?php echo esc_html( $it['title'] ); ?></h3>
								<p><?php echo esc_html( $it['desc'] ); ?></p>
							</div>
						</div>
					</a>
				<?php endforeach; ?>
			</div>

			<div class="wp-aiw-home-extras-grid">
				<div class="wp-aiw-extra-card">
					<h2>插件信息</h2>
					<div class="wp-aiw-kv">
						<div class="k">插件版本</div><div class="v"><?php echo esc_html( defined( 'WP_AIW_VERSION' ) ? WP_AIW_VERSION : '' ); ?></div>
						<div class="k">写作模型</div><div class="v"><?php echo esc_html( isset( $settings['llm_model'] ) ? (string) $settings['llm_model'] : '-' ); ?></div>
						<div class="k">图片模型</div><div class="v"><?php echo esc_html( isset( $settings['image_model'] ) ? (string) $settings['image_model'] : '-' ); ?></div>
						<div class="k">文生图接口</div><div class="v"><?php echo ! empty( $settings['image_enabled'] ) ? '<span class="dashicons dashicons-yes-alt" style="color:#00a32a;"></span> 已启用' : '<span class="dashicons dashicons-dismiss" style="color:#d63638;"></span> 未启用'; ?></div>
						<div class="k">效果演示（实际生成）</div><div class="v"><a href="https://youquso.com/blog" target="_blank" rel="noopener noreferrer">https://youquso.com/blog</a></div>
						<div class="k">使用文档</div><div class="v"><a href="https://youquso.com/products/12471.html" target="_blank" rel="noopener noreferrer">https://youquso.com/products/12471.html</a></div>
					</div>
				</div>

				<div class="wp-aiw-extra-card">
					<h2>开发信息</h2>
					<div class="wp-aiw-kv" style="margin-bottom:12px;">
						<div class="k">插件作者</div><div class="v"><a href="https://youquso.com" target="_blank" rel="noopener noreferrer">有趣搜索</a></div>
						<div class="k">插件主页</div><div class="v"><a href="https://youquso.com/products" target="_blank" rel="noopener noreferrer">https://youquso.com/products</a></div>
					</div>
					<div class="description" style="margin:0 0 10px;">
						<div>开发与反馈：</div>
						<div>（1）使用中遇到 bug 需要作者修复；</div>
						<div>（2）需要更多的功能；</div>
						<div>（3）适配更多的模型接口。</div>
						<div style="margin-top:8px;">遇到上述问题，请至官网提交一个工单，我们将尽快处理。</div>
					</div>
					<div style="margin-top:auto;">
						<a class="button button-primary" href="https://youquso.com/user-center/tickets" target="_blank" rel="noopener noreferrer">提交工单</a>
					</div>
				</div>

				<div class="wp-aiw-extra-card">
					<h2>更新公告</h2>
					<div id="wp-aiw-saas-announcements" class="description" style="margin:0;">正在加载公告...</div>
				</div>
			</div>
		</div>
		<?php
	}

	public function reorder_submenus() {
		global $submenu;
		$parent = 'wp-aiw';
		if ( empty( $submenu[ $parent ] ) || ! is_array( $submenu[ $parent ] ) ) {
			return;
		}

		// Ensure AI 写文章 is near top, and 设置 is always last.
		$items = $submenu[ $parent ];
		$home_item = null;
		$settings_item = null;
		$writer_item = null;
		$rest = array();

		foreach ( $items as $it ) {
			$slug = isset( $it[2] ) ? (string) $it[2] : '';
			if ( $slug === $parent ) {
				// Rename the auto-added parent submenu to something meaningful.
				$it[0] = '功能导航';
				$home_item = $it;
				continue;
			}
			if ( $slug === 'wp-aiw-settings' ) {
				$settings_item = $it;
				continue;
			}
			if ( $slug === 'wp-aiw-writer' ) {
				$writer_item = $it;
				continue;
			}
			$rest[] = $it;
		}

		$reordered = array();
		if ( $home_item ) {
			$reordered[] = $home_item;
		}
		if ( $writer_item ) {
			$reordered[] = $writer_item;
		}
		foreach ( $rest as $it ) {
			$reordered[] = $it;
		}
		if ( $settings_item ) {
			$reordered[] = $settings_item;
		}

		$submenu[ $parent ] = $reordered;
	}

	public function enqueue_assets( $hook ) {
		// Enqueue only on plugin pages.
		$page = isset( $_GET['page'] ) ? sanitize_key( wp_unslash( $_GET['page'] ) ) : '';
		if ( $page === 'wp-aiw' ) {
			wp_enqueue_script( 'wp-aiw-home', WP_AIW_URL . 'assets/js/home.js', array(), WP_AIW_VERSION, true );
			wp_localize_script(
				'wp-aiw-home',
				'wpAiwHome',
				array(
					'ajaxUrl' => admin_url( 'admin-ajax.php' ),
					'nonce' => wp_create_nonce( 'wp_aiw_nonce' ),
					'action' => 'wp_aiw_saas_announcements_widget',
					'limit' => 5,
				)
			);
			return;
		}

		$settings = WP_AIW_Settings::instance()->get();

		if ( $page === 'wp-aiw-runner' ) {
		$batch_size = isset( $settings['batch_size'] ) ? (int) $settings['batch_size'] : 1;
		if ( $batch_size < 1 ) {
			$batch_size = 1;
		}
		if ( $batch_size > 10 ) {
			$batch_size = 10;
		}
		wp_enqueue_script( 'wp-aiw-admin', WP_AIW_URL . 'assets/js/admin.js', array(), WP_AIW_VERSION, true );
		wp_localize_script(
			'wp-aiw-admin',
			'wpAiw',
			array(
				'ajaxUrl' => admin_url( 'admin-ajax.php' ),
				'nonce' => wp_create_nonce( 'wp_aiw_nonce' ),
				'batchSize' => $batch_size,
			)
		);
			return;
		}

		if ( $page === 'wp-aiw-featured-batch' ) {
			$batch_size = isset( $settings['batch_size'] ) ? (int) $settings['batch_size'] : 1;
			if ( $batch_size < 1 ) {
				$batch_size = 1;
			}
			if ( $batch_size > 10 ) {
				$batch_size = 10;
			}
			wp_enqueue_script( 'wp-aiw-featured-image', WP_AIW_URL . 'assets/js/featured-image.js', array(), WP_AIW_VERSION, true );
			wp_enqueue_script( 'wp-aiw-featured-batch', WP_AIW_URL . 'assets/js/featured-batch.js', array( 'wp-aiw-featured-image' ), WP_AIW_VERSION, true );
			wp_localize_script(
				'wp-aiw-featured-batch',
				'wpAiwFeaturedBatch',
				array(
					'ajaxUrl' => admin_url( 'admin-ajax.php' ),
					'nonce' => wp_create_nonce( 'wp_aiw_nonce' ),
					'batchSize' => $batch_size,
					'imageEnabled' => ! empty( $settings['image_enabled'] ) ? 1 : 0,
				)
			);
			return;
		}

		if ( $page === 'wp-aiw-settings' ) {
			wp_enqueue_script( 'wp-aiw-settings', WP_AIW_URL . 'assets/js/settings.js', array(), WP_AIW_VERSION, true );
			wp_localize_script(
				'wp-aiw-settings',
				'wpAiwSettings',
				array(
					'imageProvider' => isset( $settings['image_provider'] ) ? (string) $settings['image_provider'] : 'gemini_generatecontent',
				)
			);
			return;
		}
	}

	public function render_settings_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$settings = WP_AIW_Settings::instance()->get();
		?>
		<div class="wrap">
			<h1>AI Writer 设置</h1>
			<p>配置 LLM 接口（DeepSeek/OpenAI 兼容 chat/completions）。API Key 将保存在数据库中。</p>

			<form method="post" action="options.php">
				<?php settings_fields( 'wp_aiw_settings_group' ); ?>
				<?php $opt_name = WP_AIW_Settings::instance()->get_option_name(); ?>

				<table class="form-table" role="presentation">
					<tr>
						<th scope="row"><label for="wp-aiw-base-url">LLM Base URL</label></th>
						<td>
							<input id="wp-aiw-base-url" name="<?php echo esc_attr( $opt_name ); ?>[llm_base_url]" type="url" class="regular-text" value="<?php echo esc_attr( $settings['llm_base_url'] ); ?>" />
							<p class="description">DeepSeek（默认）：https://api.deepseek.com <br />智谱：https://open.bigmodel.cn/api/paas/v4</p>
							<p style="margin:8px 0 0;">
								<a class="button button-secondary" style="margin-right:6px;" href="<?php echo esc_url( 'https://www.bigmodel.cn/glm-coding?ic=CLSQBZJTJV' ); ?>" target="_blank" rel="noopener noreferrer">智谱平台（送免费文本和图片模型）</a>
								<a class="button button-secondary" style="margin-right:6px;" href="<?php echo esc_url( 'https://www.volcengine.com' ); ?>" target="_blank" rel="noopener noreferrer">字节火山方舟（新用户送token、模型多，推荐）</a>
								<a class="button button-secondary" href="<?php echo esc_url( 'https://platform.deepseek.com' ); ?>" target="_blank" rel="noopener noreferrer">DeepSeek平台（新用户送token）</a>
							</p>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="wp-aiw-api-key">LLM API Key</label></th>
						<td>
							<input id="wp-aiw-api-key" name="<?php echo esc_attr( $opt_name ); ?>[llm_api_key]" type="password" class="regular-text" value="<?php echo esc_attr( $settings['llm_api_key'] ); ?>" autocomplete="new-password" />
							<p class="description">此处填写各大模型平台提供的API接口的API Key（不是API ID），确保安全性，请勿泄露。</p>
							<p style="margin:8px 0 0;">
								<a class="button button-secondary" style="margin-right:6px;" href="<?php echo esc_url( 'https://platform.deepseek.com/api_keys' ); ?>" target="_blank" rel="noopener noreferrer">DeepSeek平台API Key页面</a>
								<a class="button button-secondary" style="margin-right:6px;" href="<?php echo esc_url( 'https://bigmodel.cn/usercenter/proj-mgmt/apikeys' ); ?>" target="_blank" rel="noopener noreferrer">智谱API Key页面</a>
								<a class="button button-secondary" href="<?php echo esc_url( 'https://console.volcengine.com/ark/region:ark+cn-beijing/apiKey' ); ?>" target="_blank" rel="noopener noreferrer">字节火山方舟API Key页面</a>
							</p>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="wp-aiw-model">Model</label></th>
						<td>
							<input id="wp-aiw-model" name="<?php echo esc_attr( $opt_name ); ?>[llm_model]" type="text" class="regular-text" value="<?php echo esc_attr( $settings['llm_model'] ); ?>" />
							<p class="description">DeepSeek（默认）：deepseek-chat
								<br />智谱：glm-4.7或glm-4.7-flash</p>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="wp-aiw-timeout">LLM Timeout (seconds)</label></th>
						<td>
							<input id="wp-aiw-timeout" name="<?php echo esc_attr( $opt_name ); ?>[llm_timeout_seconds]" type="number" min="5" max="600" class="small-text" value="<?php echo esc_attr( (string) $settings['llm_timeout_seconds'] ); ?>" />
							<p class="description">默认值：180，可按网络/模型接口速度调整（范围：5-600）。</p>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="wp-aiw-max-tokens">输出长度最大值（max_tokens）</label></th>
						<td>
							<input id="wp-aiw-max-tokens" name="<?php echo esc_attr( $opt_name ); ?>[llm_max_tokens]" type="number" min="0" max="65536" class="small-text" value="<?php echo esc_attr( isset( $settings['llm_max_tokens'] ) ? (string) $settings['llm_max_tokens'] : '0' ); ?>" />
							<p class="description">控制模型生成输出的最大长度（0=使用模型默认）。
								<br />当为 0 时，将仅提示模型“自行控制输出长度并优先保证 JSON 完整”，不会注入明确的输出上限数值（由模型自己决定），请参考你所用模型的输出长度限制；。
								<br />该设置对重写/写文章/URL 抓取写文章等所有文本生成均生效。
								<br />例如：deepseek-chat 默认 4K、最大 8K（填写8000）；智谱GLM-4.7 最大输出 128K（填写128000）。</p>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="wp-aiw-batch">并发数（batch_size同时处理几个任务）</label></th>
						<td>
							<input id="wp-aiw-batch" name="<?php echo esc_attr( $opt_name ); ?>[batch_size]" type="number" min="1" class="small-text" value="<?php echo esc_attr( (string) $settings['batch_size'] ); ?>" />
							<p class="description">建议从 1 开始，确认提示词与输出稳定后再逐步提高。</p>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="wp-aiw-prompt-zh">系统提示词（中文）</label></th>
						<td>
							<textarea id="wp-aiw-prompt-zh" name="<?php echo esc_attr( $opt_name ); ?>[prompt_system_zh]" class="large-text code" rows="10"><?php echo esc_textarea( $settings['prompt_system_zh'] ); ?></textarea>
							<p class="description">系统级提示词，要求模型输出严格 JSON（content_html/tags/title）。</p>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="wp-aiw-prompt-en">系统提示词（英文）</label></th>
						<td>
							<textarea id="wp-aiw-prompt-en" name="<?php echo esc_attr( $opt_name ); ?>[prompt_system_en]" class="large-text code" rows="10"><?php echo esc_textarea( $settings['prompt_system_en'] ); ?></textarea>
							<p class="description">系统级提示词，要求模型返回英文时生效。</p>
						</td>
					</tr>
				</table>

				<hr />
				<h2>AI 写文章（Writer）</h2>
				<p>配置“AI 写文章”/批量写文章的模型功能设置（独立后台功能页面使用；经典编辑器侧边栏block工具使用），与重写优化（Runner）功能使用的模型配置完全解耦，互不影响。默认开启两段式写作（先摘要清洗 URL 参考资料，再写文章），更省 token 也更稳输出；可选直接写文章但可能更慢更容易超时。</p>

				<table class="form-table" role="presentation">
					<tr>
						<th scope="row">两段式写作</th>
						<td>
							<label>
								<input name="<?php echo esc_attr( $opt_name ); ?>[writer_summary_first]" type="checkbox" value="1" <?php checked( ! empty( $settings['writer_summary_first'] ) ); ?> />
								先对 URL 参考资料做摘要/清洗，再进入写作（推荐）
							</label>
							<p class="description">开启后更省 token、更稳输出，也更不容易超时。</p>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="wp-aiw-writer-batch">批量写文章并发数</label></th>
						<td>
							<input id="wp-aiw-writer-batch" name="<?php echo esc_attr( $opt_name ); ?>[writer_batch_size]" type="number" min="1" max="10" class="small-text" value="<?php echo esc_attr( (string) $settings['writer_batch_size'] ); ?>" />
							<p class="description">仅用于“AI批量写文章”页面，与 runner 的 Batch Size（批量重写并发）区分。建议从 1 开始。</p>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="wp-aiw-url-fetch-timeout">URL Fetch Timeout (seconds)</label></th>
						<td>
							<input id="wp-aiw-url-fetch-timeout" name="<?php echo esc_attr( $opt_name ); ?>[url_fetch_timeout_seconds]" type="number" min="3" max="60" class="small-text" value="<?php echo esc_attr( (string) $settings['url_fetch_timeout_seconds'] ); ?>" />
							<p class="description">抓取 URL 参考资料的超时时间（3-60 秒）。</p>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="wp-aiw-url-fetch-max-chars">URL Fetch Max Chars</label></th>
						<td>
							<input id="wp-aiw-url-fetch-max-chars" name="<?php echo esc_attr( $opt_name ); ?>[url_fetch_max_chars]" type="number" min="1000" max="100000" class="small-text" value="<?php echo esc_attr( (string) $settings['url_fetch_max_chars'] ); ?>" />
							<p class="description">单个 URL 提取后的最大字符数（用于控制 token 与响应时间）。</p>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="wp-aiw-url-fetch-max-urls">URL Max Count</label></th>
						<td>
							<input id="wp-aiw-url-fetch-max-urls" name="<?php echo esc_attr( $opt_name ); ?>[url_fetch_max_urls]" type="number" min="1" max="10" class="small-text" value="<?php echo esc_attr( (string) $settings['url_fetch_max_urls'] ); ?>" />
							<p class="description">每次写作允许抓取的 URL 数量上限（1-10），建议值：3~5，过大可能处理超时或抓取失败。</p>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="wp-aiw-writer-prompt-zh">Writer 提示词（中文）</label></th>
						<td>
							<textarea id="wp-aiw-writer-prompt-zh" name="<?php echo esc_attr( $opt_name ); ?>[writer_prompt_system_zh]" class="large-text code" rows="10"><?php echo esc_textarea( $settings['writer_prompt_system_zh'] ); ?></textarea>
							<p class="description">系统级提示词，要求模型输出严格 JSON（title/content_html/tags）。</p>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="wp-aiw-writer-prompt-en">Writer 提示词（英文）</label></th>
						<td>
							<textarea id="wp-aiw-writer-prompt-en" name="<?php echo esc_attr( $opt_name ); ?>[writer_prompt_system_en]" class="large-text code" rows="10"><?php echo esc_textarea( $settings['writer_prompt_system_en'] ); ?></textarea>
							<p class="description">用于AI模型采用英文撰写/重写文章内容时生效。</p>
						</td>
					</tr>
				</table>

				<hr />
				<h2>文生图（Image）</h2>
				<p>用于生成文章插图（与写文章/重写的 LLM 配置完全解耦）。默认关闭。</p>

				<table class="form-table" role="presentation">
					<tr>
						<th scope="row">启用文生图</th>
						<td>
							<label>
								<input name="<?php echo esc_attr( $opt_name ); ?>[image_enabled]" type="checkbox" value="1" <?php checked( ! empty( $settings['image_enabled'] ) ); ?> />
								启用文生图接口（仅允许已配置时调用）
							</label>
							<p class="description">开启后将启用所有图片相关能力（生成/入库/插入正文/设为特色图），未正确配置模型参数前请勿开启。</p>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="wp-aiw-image-provider">Provider</label></th>
						<td>
							<select id="wp-aiw-image-provider" name="<?php echo esc_attr( $opt_name ); ?>[image_provider]">
								<option value="zhipu_glm_image" <?php selected( $settings['image_provider'], 'zhipu_glm_image' ); ?>>智谱 GLM-Image</option>
								<option value="openai_compatible_images" <?php selected( $settings['image_provider'], 'openai_compatible_images' ); ?>>豆包/OpenAI-compatible</option>
								<option value="gemini_generatecontent" <?php selected( $settings['image_provider'], 'gemini_generatecontent' ); ?>>Gemini</option>
							</select>
							<p class="description">不同模型/平台协议不同，不一定兼容返回的图片数据格式。
								<br />智谱 GLM-Image 使用 x-authorization + /v4/images/generations（Zhipu 默认返回 url）。
								<br />OpenAI-compatible 则使用 /images/generations + Bearer Token，支持返回 url 或 base64（需模型支持）。
								<br />Gemini 使用 x-goog-api-key + /v1beta/models/{model}:generateContent。
								<br />字节豆包/OpenAI-compatible 额外支持 response_format 与 Extra Params（字节火山Ark豆包模型API返回兼容）。</p>
							<p style="margin:8px 0 0;">
								<a class="button button-secondary" style="margin-right:6px;" href="<?php echo esc_url( 'https://www.bigmodel.cn/glm-coding?ic=CLSQBZJTJV' ); ?>" target="_blank" rel="noopener noreferrer">智谱平台（送免费文本和图片模型）</a>
								<a class="button button-secondary" href="<?php echo esc_url( 'https://www.volcengine.com' ); ?>" target="_blank" rel="noopener noreferrer">字节火山方舟（新用户送token、模型多，推荐）</a>
							</p>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="wp-aiw-image-base-url">Image Base URL</label></th>
						<td>
							<input id="wp-aiw-image-base-url" name="<?php echo esc_attr( $opt_name ); ?>[image_base_url]" type="url" class="regular-text" value="<?php echo esc_attr( $settings['image_base_url'] ); ?>" />
							<br class="description">
							<br />智谱 glm-image：https://open.bigmodel.cn/api/paas/v4
							<br />字节火山方舟 ：https://ark.cn-beijing.volces.com/api/v3
							<br />Gemini：https://generativelanguage.googleapis.com
							<br />OpenAI-compatible类模型：https://api.example.com（将调用 /images/generations）
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="wp-aiw-image-api-key">Image API Key</label></th>
						<td>
							<input id="wp-aiw-image-api-key" name="<?php echo esc_attr( $opt_name ); ?>[image_api_key]" type="password" class="regular-text" value="<?php echo esc_attr( $settings['image_api_key'] ); ?>" autocomplete="new-password" />
							<p class="description">
								此处填写各大模型平台提供的API接口的API Key（不是API ID），确保安全性，请勿泄露。
							</p>
							<p style="margin:8px 0 0;">
								<a class="button button-secondary" style="margin-right:6px;" href="<?php echo esc_url( 'https://bigmodel.cn/usercenter/proj-mgmt/apikeys' ); ?>" target="_blank" rel="noopener noreferrer">智谱API Key获取地址</a>
								<a class="button button-secondary" href="<?php echo esc_url( 'https://console.volcengine.com/ark/region:ark+cn-beijing/apiKey' ); ?>" target="_blank" rel="noopener noreferrer">字节火山方舟API Key获取地址</a>
							</p>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="wp-aiw-image-model">Image Model</label></th>
						<td>
							<input id="wp-aiw-image-model" name="<?php echo esc_attr( $opt_name ); ?>[image_model]" type="text" class="regular-text" value="<?php echo esc_attr( $settings['image_model'] ); ?>" placeholder="例如：nano-banana" />
							<p class="description">
								智谱模型（默认，价格适中，用于文章特色图足够）：glm-image<br />
								字节豆包大模型（火山平台，贵，生图快）：doubao-seedream-4-5-251128<br />
								Gemini模型（需要能连外网）：gemini-2.5-flash-image 或  <br />
								OpenAI-compatible 则填写对应的 image model 名称（需要查文档）
							</p>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="wp-aiw-image-timeout">Image Timeout (seconds)</label></th>
						<td>
							<input id="wp-aiw-image-timeout" name="<?php echo esc_attr( $opt_name ); ?>[image_timeout_seconds]" type="number" min="10" max="600" class="small-text" value="<?php echo esc_attr( (string) $settings['image_timeout_seconds'] ); ?>" />
							<p class="description">图生成通常更慢，建议 60-300 秒，一般字节火山较快（基本稳定在30s），但价格比智谱贵，智谱平均生成一张图在30~50s。</p>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="wp-aiw-image-size">Image Size</label></th>
						<td>
							<input id="wp-aiw-image-size" name="<?php echo esc_attr( $opt_name ); ?>[image_size]" type="text" class="regular-text" value="<?php echo esc_attr( $settings['image_size'] ); ?>" placeholder="1024x1024" />
							<p id="wp-aiw-image-size-note" class="description">智谱：1280x1280（默认）
								<br />字节豆包（火山ark）：2k
								<br />不同供应商支持的尺寸不同，如果更换其他的请查阅文档。</p>
						</td>
					</tr>
					<tr id="wp-aiw-image-openai-only-response-format">
						<th scope="row"><label for="wp-aiw-image-response-format">Response Format</label></th>
						<td>
							<select id="wp-aiw-image-response-format" name="<?php echo esc_attr( $opt_name ); ?>[image_response_format]">
								<option value="b64_json" <?php selected( $settings['image_response_format'], 'b64_json' ); ?>>b64_json（返回 base64，体积大）</option>
								<option value="url" <?php selected( $settings['image_response_format'], 'url' ); ?>>url（返回链接，推荐：火山 Ark 示例）</option>
							</select>
							<p class="description">仅 OpenAI-compatible Provider 生效。若供应商不支持 b64_json，请改为 url。</p>
						</td>
					</tr>
					<tr id="wp-aiw-image-openai-only-extra-params">
						<th scope="row"><label for="wp-aiw-image-extra-params">Extra Params (JSON)</label></th>
						<td>
							<textarea id="wp-aiw-image-extra-params" name="<?php echo esc_attr( $opt_name ); ?>[image_extra_params_json]" class="large-text code" rows="4" placeholder='{"watermark":true,"stream":false,"sequential_image_generation":"disabled"}'><?php echo esc_textarea( isset( $settings['image_extra_params_json'] ) ? (string) $settings['image_extra_params_json'] : '' ); ?></textarea>
							<p class="description">仅 OpenAI-compatible Provider 生效。用于透传供应商特有字段（必须是 JSON 对象）。火山 Ark（Doubao Seedream）常用：watermark/stream/sequential_image_generation。</p>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="wp-aiw-image-max">每篇文章最多生成多少张图片</label></th>
						<td>
							<input id="wp-aiw-image-max" name="<?php echo esc_attr( $opt_name ); ?>[image_max_per_post]" type="number" min="0" max="20" class="small-text" value="<?php echo esc_attr( (string) $settings['image_max_per_post'] ); ?>" />
							<p class="description">强烈建议：1
								<br />0 表示不限制（不推荐）。</p>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="wp-aiw-image-prompt-zh">Image Prompt Prefix（中文）</label></th>
						<td>
							<textarea id="wp-aiw-image-prompt-zh" name="<?php echo esc_attr( $opt_name ); ?>[image_prompt_prefix_zh]" class="large-text code" rows="4"><?php echo esc_textarea( $settings['image_prompt_prefix_zh'] ); ?></textarea>
							<p class="description">用于把“标题/要点”转换成可用于文生图的英文 prompt。</p>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="wp-aiw-image-prompt-en">Image Prompt Prefix（英文）</label></th>
						<td>
							<textarea id="wp-aiw-image-prompt-en" name="<?php echo esc_attr( $opt_name ); ?>[image_prompt_prefix_en]" class="large-text code" rows="4"><?php echo esc_textarea( $settings['image_prompt_prefix_en'] ); ?></textarea>
						</td>
					</tr>
				</table>

				<?php submit_button( '保存设置' ); ?>
			</form>
		</div>
		<?php
	}

	public function render_runner_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}
		$categories = get_categories( array( 'hide_empty' => false ) );
		$tags = get_terms( array( 'taxonomy' => 'post_tag', 'hide_empty' => false ) );
		$authors = get_users( array(
			'orderby' => 'display_name',
			'order' => 'ASC',
			'fields' => array( 'ID', 'display_name' ),
		) );
		?>
		<div class="wrap">
			<style>
				/* Runner page small UI tweaks */
				.wp-aiw-runner-filters .wp-aiw-filter {
					flex: 0 1 170px;
					min-width: 170px;
				}
				.wp-aiw-runner-filters .wp-aiw-filter-cat {
					flex: 0 1 320px;
					min-width: 320px;
				}
				.wp-aiw-runner-filters .wp-aiw-filter select,
				.wp-aiw-runner-filters .wp-aiw-filter input[type="date"] {
					width: 170px;
					max-width: 100%;
				}
				.wp-aiw-runner-filters .wp-aiw-filter-cat select {
					width: 200px;
					max-width: 100%;
				}
				.wp-aiw-runner-filters .wp-aiw-filter-wide {
					flex: 0 1 220px;
					min-width: 220px;
				}
				.wp-aiw-runner-filters .wp-aiw-filter-wide select {
					width: 220px;
					max-width: 100%;
				}
				.wp-aiw-runner-filters .wp-aiw-inline-row {
					display: flex;
					align-items: center;
					gap: 10px;
					flex-wrap: wrap;
				}
				.wp-aiw-runner-filters .wp-aiw-inline-row label {
					margin: 0;
				}
			</style>
			<h1>AI Writer / 重写文章</h1>
			<p>点击开始后，插件将并发处理文章（并发数=设置里的 Batch Size），调用 LLM 重写正文与补充标签，并写回原文章。</p>
			<p><strong>提示：</strong>建议先在测试环境验证提示词与输出格式，再用于生产。</p>

			<div style="margin: 12px 0; padding: 12px; background: #fff; border: 1px solid #dcdcde; border-radius: 6px;">
				<div style="display:flex; align-items:center; justify-content:space-between; gap: 12px; flex-wrap: wrap;">
					<div>
						<strong>运行控制</strong>
						<p class="description" style="margin: 6px 0 0;">点击开始后将按并发数持续拉取并处理“下一篇”；可随时停止。</p>
					</div>
					<div style="display:flex; gap: 8px; align-items:center;">
						<button class="button button-primary" id="wp-aiw-start">开始</button>
						<button class="button" id="wp-aiw-resume" style="display:none;">继续上次运行</button>
						<button class="button" id="wp-aiw-stop" disabled>停止</button>
						<button class="button" id="wp-aiw-clear" type="button">清除任务</button>
					</div>
				</div>
				<div style="margin-top: 10px; display:flex; gap: 12px; align-items:center; flex-wrap: wrap;">
					<label style="display: inline-flex; align-items: center; gap: 6px;">
						<input type="checkbox" id="wp-aiw-skip-failed" checked />
						<span>失败时跳过（仅本次运行；方便下次再次尝试）</span>
					</label>
					<label style="display: inline-flex; align-items: center; gap: 6px;">
						<input type="checkbox" id="wp-aiw-force-rewrite" />
						<span>强制重写（包含已处理文章）</span>
					</label>
				</div>

				<div id="wp-aiw-status" style="margin-top: 10px; padding: 10px 12px; background: #f6f7f7; border: 1px solid #dcdcde; border-radius: 6px;">待开始</div>
			</div>

			<div style="margin: 12px 0; padding: 12px; background: #fff; border: 1px solid #dcdcde; border-radius: 6px;">
				<strong>筛选规则</strong>
				<div class="wp-aiw-runner-filters" style="margin-top: 10px; display:flex; flex-wrap: wrap; gap: 10px; align-items: flex-end;">
					<div class="wp-aiw-filter wp-aiw-filter-cat">
						<label for="wp-aiw-category"><span style="display:block; margin-bottom:4px;">分类</span></label>
						<div class="wp-aiw-inline-row">
							<select id="wp-aiw-category" class="regular-text">
								<option value="0">不限制</option>
								<?php foreach ( $categories as $c ) : ?>
									<option value="<?php echo esc_attr( (string) $c->term_id ); ?>"><?php echo esc_html( $c->name ); ?></option>
								<?php endforeach; ?>
							</select>
							<label style="display: inline-flex; align-items: center; gap: 6px; white-space: nowrap;">
								<input type="checkbox" id="wp-aiw-category-include-children" checked />
								<span>包含子分类</span>
							</label>
						</div>
					</div>

					<div class="wp-aiw-filter">
						<label for="wp-aiw-tag"><span style="display:block; margin-bottom:4px;">标签</span></label>
						<select id="wp-aiw-tag" class="regular-text">
							<option value="0">不限制</option>
							<?php if ( ! is_wp_error( $tags ) && is_array( $tags ) ) : ?>
								<?php foreach ( $tags as $t ) : ?>
									<option value="<?php echo esc_attr( (string) $t->term_id ); ?>"><?php echo esc_html( $t->name ); ?></option>
								<?php endforeach; ?>
							<?php endif; ?>
						</select>
					</div>

					<div class="wp-aiw-filter">
						<label for="wp-aiw-date-after"><span style="display:block; margin-bottom:4px;">时间范围（起）</span></label>
						<input id="wp-aiw-date-after" type="date" class="regular-text" />
					</div>

					<div class="wp-aiw-filter">
						<label for="wp-aiw-date-before"><span style="display:block; margin-bottom:4px;">时间范围（止）</span></label>
						<input id="wp-aiw-date-before" type="date" class="regular-text" />
					</div>

					<div class="wp-aiw-filter">
						<label for="wp-aiw-author"><span style="display:block; margin-bottom:4px;">作者</span></label>
						<select id="wp-aiw-author" class="regular-text">
							<option value="0">不限制</option>
							<?php if ( is_array( $authors ) ) : ?>
								<?php foreach ( $authors as $u ) : ?>
									<option value="<?php echo esc_attr( (string) $u->ID ); ?>"><?php echo esc_html( $u->display_name ); ?></option>
								<?php endforeach; ?>
							<?php endif; ?>
						</select>
					</div>

					<div class="wp-aiw-filter">
						<label for="wp-aiw-order"><span style="display:block; margin-bottom:4px;">排序</span></label>
						<select id="wp-aiw-order" class="regular-text">
							<option value="oldest">最旧优先（ID 升序）</option>
							<option value="modified_desc">最近更新优先（modified 降序）</option>
						</select>
					</div>

					<div class="wp-aiw-filter-wide">
						<label for="wp-aiw-language"><span style="display:block; margin-bottom:4px;">输出语言</span></label>
						<select id="wp-aiw-language" class="regular-text">
							<option value="zh">中文</option>
							<option value="en">English</option>
						</select>
						<label style="display: inline-flex; align-items: center; gap: 6px; margin-top: 6px;">
							<input type="checkbox" id="wp-aiw-rewrite-title" />
							<span>重写标题（默认不重写）</span>
						</label>
					</div>
				</div>
				<p class="description" style="margin: 10px 0 0;">筛选仅影响“下一篇文章怎么选”。默认会跳过已处理文章（自定义字段 wp_aiw_rewritten=1）；勾选“强制重写”可重新处理。</p>
			</div>

			<div id="wp-aiw-log" style="max-height: 420px; overflow: auto; background: #111827; color: #e5e7eb; padding: 12px; border-radius: 8px; font-family: ui-monospace, SFMono-Regular, Menlo, Monaco, Consolas, 'Liberation Mono', 'Courier New', monospace; font-size: 12px; white-space: pre-wrap; word-break: break-word; line-height: 1.55;"></div>
		</div>
		<?php
	}

	public function ajax_process_next() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => '无权限' ), 403 );
		}
		check_ajax_referer( 'wp_aiw_nonce', 'nonce' );

		$category_id = isset( $_POST['category_id'] ) ? (int) $_POST['category_id'] : 0;
		$tag_id      = isset( $_POST['tag_id'] ) ? (int) $_POST['tag_id'] : 0;
		$author_id   = isset( $_POST['author_id'] ) ? (int) $_POST['author_id'] : 0;
		$include_children = isset( $_POST['category_include_children'] ) ? (bool) (int) $_POST['category_include_children'] : true;
		$date_after  = isset( $_POST['date_after'] ) ? (string) wp_unslash( $_POST['date_after'] ) : '';
		$date_before = isset( $_POST['date_before'] ) ? (string) wp_unslash( $_POST['date_before'] ) : '';
		$order_mode  = isset( $_POST['order_mode'] ) ? (string) wp_unslash( $_POST['order_mode'] ) : 'oldest';
		$language    = isset( $_POST['language'] ) ? (string) wp_unslash( $_POST['language'] ) : 'zh';
		$rewrite_title = isset( $_POST['rewrite_title'] ) ? (bool) (int) $_POST['rewrite_title'] : false;
		$skip_failed = isset( $_POST['skip_failed'] ) ? (bool) (int) $_POST['skip_failed'] : true;
		$force_rewrite = isset( $_POST['force_rewrite'] ) ? (bool) (int) $_POST['force_rewrite'] : false;
		$exclude_ids_raw = isset( $_POST['exclude_ids'] ) ? (string) wp_unslash( $_POST['exclude_ids'] ) : '';
		$date_after  = trim( $date_after );
		$date_before = trim( $date_before );
		$order_mode  = trim( $order_mode );
		$language    = trim( $language );
		if ( $language !== 'en' ) {
			$language = 'zh';
		}
		if ( $order_mode !== 'modified_desc' ) {
			$order_mode = 'oldest';
		}
		// 日期只接受 YYYY-MM-DD
		if ( $date_after !== '' && ! preg_match( '/^\d{4}-\d{2}-\d{2}$/', $date_after ) ) {
			$date_after = '';
		}
		if ( $date_before !== '' && ! preg_match( '/^\d{4}-\d{2}-\d{2}$/', $date_before ) ) {
			$date_before = '';
		}

		$settings = WP_AIW_Settings::instance()->get();
		$rewriter = new WP_AIW_Rewriter();
		$lock_key = '_wp_aiw_in_progress';
		$lock_ttl_seconds = 30 * MINUTE_IN_SECONDS;
		$criteria = array(
			'category_id' => max( 0, $category_id ),
			'tag_id' => max( 0, $tag_id ),
			'author_id' => max( 0, $author_id ),
			'category_include_children' => $include_children,
			'date_after' => $date_after,
			'date_before' => $date_before,
			'order_mode' => $order_mode,
			'language' => $language,
			'rewrite_title' => $rewrite_title,
			'skip_failed' => $skip_failed,
			'force_rewrite' => $force_rewrite,
			'exclude_ids' => $exclude_ids_raw,
		);

		// 防止并发：多个 AJAX 同时选中同一篇文章。
		$exclude_ids = array();
		if ( $exclude_ids_raw !== '' ) {
			$parts = preg_split( '/[^0-9]+/', $exclude_ids_raw );
			if ( is_array( $parts ) ) {
				foreach ( $parts as $p ) {
					$id = (int) $p;
					if ( $id > 0 ) {
						$exclude_ids[] = $id;
					}
				}
			}
		}
		$exclude_ids = array_values( array_unique( $exclude_ids ) );

		$next_id = 0;
		for ( $attempt = 0; $attempt < 10; $attempt++ ) {
			$criteria['exclude_ids'] = implode( ',', $exclude_ids );
			$candidate = $rewriter->find_next_post_id( $criteria );
			if ( $candidate <= 0 ) {
				break;
			}

			// Acquire lock (unique meta). If exists and is stale, clear and retry once.
			$locked = add_post_meta( $candidate, $lock_key, time(), true );
			if ( ! $locked ) {
				$locked_at = (int) get_post_meta( $candidate, $lock_key, true );
				if ( $locked_at > 0 && ( time() - $locked_at ) > $lock_ttl_seconds ) {
					delete_post_meta( $candidate, $lock_key );
					$locked = add_post_meta( $candidate, $lock_key, time(), true );
				}
			}

			if ( $locked ) {
				$next_id = (int) $candidate;
				break;
			}

			$exclude_ids[] = (int) $candidate;
			$exclude_ids = array_values( array_unique( $exclude_ids ) );
			if ( count( $exclude_ids ) > 500 ) {
				$exclude_ids = array_slice( $exclude_ids, 0, 500 );
			}
		}

		if ( $next_id <= 0 ) {
			wp_send_json_success( array( 'done' => true, 'message' => '没有更多未处理文章了' ) );
		}

		$runner_options = array(
			'language' => $language,
			'rewrite_title' => $rewrite_title,
		);
		$result = $rewriter->rewrite_post( $next_id, $settings, $runner_options );
		// Always release lock.
		delete_post_meta( $next_id, $lock_key );
		if ( empty( $result['ok'] ) ) {
			$message = isset( $result['error'] ) ? (string) $result['error'] : '处理失败';
			$raw = isset( $result['raw'] ) ? $result['raw'] : null;
			wp_send_json_success( array(
				'done' => false,
				'ok' => false,
				'post_id' => $next_id,
				'post_title' => get_the_title( $next_id ),
				'message' => $message,
				'raw' => $raw,
				'marked_failed' => 0,
			) );
		}

		wp_send_json_success( array(
			'done' => false,
			'ok' => true,
			'post_id' => $next_id,
			'post_title' => get_the_title( $next_id ),
			'tags_added' => isset( $result['tags_added'] ) ? $result['tags_added'] : array(),
			'message' => '已重写并写回',
		) );
	}

	public function render_featured_batch_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}
		$categories = get_categories( array( 'hide_empty' => false ) );
		$tags = get_terms( array( 'taxonomy' => 'post_tag', 'hide_empty' => false ) );
		$authors = get_users( array(
			'orderby' => 'display_name',
			'order' => 'ASC',
			'fields' => array( 'ID', 'display_name' ),
		) );
		?>
		<div class="wrap">
			<style>
				.wp-aiw-runner-filters .wp-aiw-filter { flex: 0 1 170px; min-width: 170px; }
				.wp-aiw-runner-filters .wp-aiw-filter-cat { flex: 0 1 320px; min-width: 320px; }
				.wp-aiw-runner-filters .wp-aiw-filter select,
				.wp-aiw-runner-filters .wp-aiw-filter input[type="date"] { width: 170px; max-width: 100%; }
				.wp-aiw-runner-filters .wp-aiw-filter-cat select { width: 200px; max-width: 100%; }
				.wp-aiw-runner-filters .wp-aiw-filter-wide { flex: 0 1 220px; min-width: 220px; }
				.wp-aiw-runner-filters .wp-aiw-filter-wide select { width: 220px; max-width: 100%; }
				.wp-aiw-runner-filters .wp-aiw-inline-row { display:flex; align-items:center; gap:10px; flex-wrap:wrap; }
				.wp-aiw-runner-filters .wp-aiw-inline-row label { margin: 0; }
			</style>
			<h1>AI Writer / 批量AI特色图生成</h1>
			<p>按筛选范围批量生成并设置文章特色图（并发数=设置里的 Batch Size）。</p>
			<p class="description" style="margin-top: 6px;">默认会<strong>逐篇</strong>使用该文章的<strong>标题</strong>（并结合正文摘录）来生成特色图；下面的 requirements/styleHint/尺寸 仅用于在此基础上做“统一风格/额外限制”。不填也可以正常生成。</p>

			<div style="margin: 12px 0; padding: 12px; background: #fff; border: 1px solid #dcdcde; border-radius: 6px;">
				<div style="display:flex; align-items:center; justify-content:space-between; gap: 12px; flex-wrap: wrap;">
					<div>
						<strong>运行控制</strong>
						<p class="description" style="margin: 6px 0 0;">开始后将并发持续拉取并处理“下一篇”；可随时停止，刷新后可继续上次运行。</p>
					</div>
					<div style="display:flex; gap: 8px; align-items:center;">
						<button class="button button-primary" id="wp-aiw-fi-start">开始</button>
						<button class="button" id="wp-aiw-fi-resume" style="display:none;">继续上次运行</button>
						<button class="button" id="wp-aiw-fi-stop" disabled>停止</button>
						<button class="button" id="wp-aiw-fi-clear" type="button">清除任务</button>
					</div>
				</div>

				<div style="margin-top: 12px; display:grid; grid-template-columns: 1fr 1fr; gap: 12px;">
					<div style="grid-column: 1 / -1;">
						<label for="wp-aiw-fi-req"><strong>自定义要求（requirements，可选）</strong></label>
						<textarea id="wp-aiw-fi-req" class="large-text" rows="3" placeholder="例如：扁平插画风、科技感、无文字、蓝色主色调、留白多"></textarea>
						<p class="description" style="margin: 6px 0 0;">可留空。填写后会作为“全局附加要求”参与每篇文章的图片 prompt 生成；建议简短，强调风格/构图/不要文字等规则。</p>
					</div>
					<div>
						<label for="wp-aiw-fi-style"><strong>风格提示（styleHint，可选）</strong></label>
						<input id="wp-aiw-fi-style" type="text" class="regular-text" style="width:100%;" placeholder="例如：isometric, minimal, no text" />
						<p class="description" style="margin: 6px 0 0;">可留空。更短的风格关键词；与 requirements 可同时使用。</p>
					</div>
					<div>
						<label for="wp-aiw-fi-size"><strong>尺寸（size，可选）</strong></label>
						<input id="wp-aiw-fi-size" type="text" class="regular-text" style="width:100%;" placeholder="例如：1024x1024（留空=使用设置页 image_size）" />
						<p class="description" style="margin: 6px 0 0;">可留空。不同 Provider 支持的尺寸不同；留空则使用设置页的 Image Size。</p>
					</div>
				</div>
				<div style="margin-top: 10px; display:flex; gap: 12px; align-items:flex-start; flex-wrap: wrap;">
					<div style="min-width: 320px;">
						<div><strong>生成模式</strong></div>
						<label style="display:block; margin-top: 6px;">
							<input type="radio" name="wp-aiw-fi-mode" value="missing_only" checked />
							仅为范围内没有特色图的文章生成（默认）
						</label>
						<label style="display:block; margin-top: 6px;">
							<input type="radio" name="wp-aiw-fi-mode" value="regenerate_all" />
							重新生成范围内所有文章特色图（先删除原特色图及媒体库文件，再重新生成）
						</label>
						<p class="description" style="margin: 6px 0 0;">提示：默认“仅缺失”会跳过已有特色图文章；“重新生成所有”会删除原特色图附件（媒体库文件），若该附件被其他文章/页面引用，会一并受影响。</p>
					</div>
					<div>
						<label style="display: inline-flex; align-items: center; gap: 6px; margin-top: 22px;">
							<input type="checkbox" id="wp-aiw-fi-skip-failed" checked />
							<span>失败时跳过（仅本次运行；方便下次再次尝试）</span>
						</label>
					</div>
				</div>

				<div id="wp-aiw-fi-status" style="margin-top: 10px; padding: 10px 12px; background: #f6f7f7; border: 1px solid #dcdcde; border-radius: 6px;">待开始</div>
			</div>

			<div style="margin: 12px 0; padding: 12px; background: #fff; border: 1px solid #dcdcde; border-radius: 6px;">
				<strong>筛选规则</strong>
				<div class="wp-aiw-runner-filters" style="margin-top: 10px; display:flex; flex-wrap: wrap; gap: 10px; align-items: flex-end;">
					<div class="wp-aiw-filter wp-aiw-filter-cat">
						<label for="wp-aiw-fi-category"><span style="display:block; margin-bottom:4px;">分类</span></label>
						<div class="wp-aiw-inline-row">
							<select id="wp-aiw-fi-category" class="regular-text">
								<option value="0">不限制</option>
								<?php foreach ( $categories as $c ) : ?>
									<option value="<?php echo esc_attr( (string) $c->term_id ); ?>"><?php echo esc_html( $c->name ); ?></option>
								<?php endforeach; ?>
							</select>
							<label style="display: inline-flex; align-items: center; gap: 6px; white-space: nowrap;">
								<input type="checkbox" id="wp-aiw-fi-category-include-children" checked />
								<span>包含子分类</span>
							</label>
						</div>
					</div>

					<div class="wp-aiw-filter">
						<label for="wp-aiw-fi-tag"><span style="display:block; margin-bottom:4px;">标签</span></label>
						<select id="wp-aiw-fi-tag" class="regular-text">
							<option value="0">不限制</option>
							<?php if ( ! is_wp_error( $tags ) && is_array( $tags ) ) : ?>
								<?php foreach ( $tags as $t ) : ?>
									<option value="<?php echo esc_attr( (string) $t->term_id ); ?>"><?php echo esc_html( $t->name ); ?></option>
								<?php endforeach; ?>
							<?php endif; ?>
						</select>
					</div>

					<div class="wp-aiw-filter">
						<label for="wp-aiw-fi-date-after"><span style="display:block; margin-bottom:4px;">时间范围（起）</span></label>
						<input id="wp-aiw-fi-date-after" type="date" class="regular-text" />
					</div>

					<div class="wp-aiw-filter">
						<label for="wp-aiw-fi-date-before"><span style="display:block; margin-bottom:4px;">时间范围（止）</span></label>
						<input id="wp-aiw-fi-date-before" type="date" class="regular-text" />
					</div>

					<div class="wp-aiw-filter">
						<label for="wp-aiw-fi-author"><span style="display:block; margin-bottom:4px;">作者</span></label>
						<select id="wp-aiw-fi-author" class="regular-text">
							<option value="0">不限制</option>
							<?php if ( is_array( $authors ) ) : ?>
								<?php foreach ( $authors as $u ) : ?>
									<option value="<?php echo esc_attr( (string) $u->ID ); ?>"><?php echo esc_html( $u->display_name ); ?></option>
								<?php endforeach; ?>
							<?php endif; ?>
						</select>
					</div>

					<div class="wp-aiw-filter">
						<label for="wp-aiw-fi-order"><span style="display:block; margin-bottom:4px;">排序</span></label>
						<select id="wp-aiw-fi-order" class="regular-text">
							<option value="oldest">最旧优先（ID 升序）</option>
							<option value="modified_desc">最近更新优先（modified 降序）</option>
						</select>
					</div>

					<div class="wp-aiw-filter-wide">
						<label for="wp-aiw-fi-language"><span style="display:block; margin-bottom:4px;">输入语言</span></label>
						<select id="wp-aiw-fi-language" class="regular-text">
							<option value="zh">中文</option>
							<option value="en">English</option>
						</select>
						<p class="description" style="margin: 6px 0 0;">用于生成图片 prompt 的输入语言（标题/requirements 的理解）；最终用于文生图的 prompt 会被转换为英文。</p>
					</div>
				</div>
				<p class="description" style="margin: 10px 0 0;">筛选仅影响“下一篇文章怎么选”。默认按每篇文章标题（并结合正文摘录）生成；在“仅缺失模式”下会跳过已有特色图的文章。</p>
			</div>

			<div id="wp-aiw-fi-log" style="max-height: 420px; overflow: auto; background: #111827; color: #e5e7eb; padding: 12px; border-radius: 8px; font-family: ui-monospace, SFMono-Regular, Menlo, Monaco, Consolas, 'Liberation Mono', 'Courier New', monospace; font-size: 12px; white-space: pre-wrap; word-break: break-word; line-height: 1.55;"></div>
		</div>
		<?php
	}

	public function ajax_featured_batch_next() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => '无权限' ), 403 );
		}
		check_ajax_referer( 'wp_aiw_nonce', 'nonce' );

		$category_id = isset( $_POST['category_id'] ) ? (int) $_POST['category_id'] : 0;
		$tag_id      = isset( $_POST['tag_id'] ) ? (int) $_POST['tag_id'] : 0;
		$author_id   = isset( $_POST['author_id'] ) ? (int) $_POST['author_id'] : 0;
		$include_children = isset( $_POST['category_include_children'] ) ? (bool) (int) $_POST['category_include_children'] : true;
		$date_after  = isset( $_POST['date_after'] ) ? (string) wp_unslash( $_POST['date_after'] ) : '';
		$date_before = isset( $_POST['date_before'] ) ? (string) wp_unslash( $_POST['date_before'] ) : '';
		$order_mode  = isset( $_POST['order_mode'] ) ? (string) wp_unslash( $_POST['order_mode'] ) : 'oldest';
		$language    = isset( $_POST['language'] ) ? (string) wp_unslash( $_POST['language'] ) : 'zh';
		$mode        = isset( $_POST['mode'] ) ? (string) wp_unslash( $_POST['mode'] ) : 'missing_only';
		$skip_failed = isset( $_POST['skip_failed'] ) ? (bool) (int) $_POST['skip_failed'] : true;
		$requirements = isset( $_POST['requirements'] ) ? (string) wp_unslash( $_POST['requirements'] ) : '';
		$style_hint = isset( $_POST['style_hint'] ) ? (string) wp_unslash( $_POST['style_hint'] ) : '';
		$size = isset( $_POST['size'] ) ? (string) wp_unslash( $_POST['size'] ) : '';
		$exclude_ids_raw = isset( $_POST['exclude_ids'] ) ? (string) wp_unslash( $_POST['exclude_ids'] ) : '';
		$date_after  = trim( $date_after );
		$date_before = trim( $date_before );
		$order_mode  = trim( $order_mode );
		$language    = trim( $language );
		$mode        = trim( $mode );
		$requirements = is_string( $requirements ) ? trim( $requirements ) : '';
		$style_hint = is_string( $style_hint ) ? trim( $style_hint ) : '';
		$size = is_string( $size ) ? trim( $size ) : '';
		if ( $language !== 'en' ) {
			$language = 'zh';
		}
		if ( $order_mode !== 'modified_desc' ) {
			$order_mode = 'oldest';
		}
		if ( $mode !== 'regenerate_all' ) {
			$mode = 'missing_only';
		}
		if ( $requirements !== '' ) {
			if ( function_exists( 'mb_substr' ) ) {
				$requirements = mb_substr( $requirements, 0, 4000 );
			} else {
				$requirements = substr( $requirements, 0, 4000 );
			}
		}
		if ( $style_hint !== '' ) {
			if ( function_exists( 'mb_substr' ) ) {
				$style_hint = mb_substr( $style_hint, 0, 200 );
			} else {
				$style_hint = substr( $style_hint, 0, 200 );
			}
		}
		$size = sanitize_text_field( $size );
		if ( $date_after !== '' && ! preg_match( '/^\d{4}-\d{2}-\d{2}$/', $date_after ) ) {
			$date_after = '';
		}
		if ( $date_before !== '' && ! preg_match( '/^\d{4}-\d{2}-\d{2}$/', $date_before ) ) {
			$date_before = '';
		}

		$settings = WP_AIW_Settings::instance()->get();
		if ( empty( $settings['image_enabled'] ) ) {
			wp_send_json_error( array( 'message' => '图片功能未启用（image_enabled=0）' ), 400 );
		}

		$criteria = array(
			'category_id' => max( 0, $category_id ),
			'tag_id' => max( 0, $tag_id ),
			'author_id' => max( 0, $author_id ),
			'category_include_children' => $include_children,
			'date_after' => $date_after,
			'date_before' => $date_before,
			'order_mode' => $order_mode,
			'mode' => $mode,
			'exclude_ids' => $exclude_ids_raw,
		);

		$lock_key = '_wp_aiw_featured_in_progress';
		$lock_ttl_seconds = 30 * MINUTE_IN_SECONDS;

		$exclude_ids = array();
		if ( $exclude_ids_raw !== '' ) {
			$parts = preg_split( '/[^0-9]+/', $exclude_ids_raw );
			if ( is_array( $parts ) ) {
				foreach ( $parts as $p ) {
					$id = (int) $p;
					if ( $id > 0 ) {
						$exclude_ids[] = $id;
					}
				}
			}
		}
		$exclude_ids = array_values( array_unique( $exclude_ids ) );

		$runner = class_exists( 'WP_AIW_Featured_Batch' ) ? new WP_AIW_Featured_Batch() : null;
		if ( ! $runner ) {
			wp_send_json_error( array( 'message' => '缺少批量特色图组件' ), 500 );
		}

		$next_id = 0;
		for ( $attempt = 0; $attempt < 10; $attempt++ ) {
			$criteria['exclude_ids'] = implode( ',', $exclude_ids );
			$candidate = $runner->find_next_post_id( $criteria );
			if ( $candidate <= 0 ) {
				break;
			}

			$locked = add_post_meta( $candidate, $lock_key, time(), true );
			if ( ! $locked ) {
				$locked_at = (int) get_post_meta( $candidate, $lock_key, true );
				if ( $locked_at > 0 && ( time() - $locked_at ) > $lock_ttl_seconds ) {
					delete_post_meta( $candidate, $lock_key );
					$locked = add_post_meta( $candidate, $lock_key, time(), true );
				}
			}

			if ( $locked ) {
				$next_id = (int) $candidate;
				break;
			}

			$exclude_ids[] = (int) $candidate;
			$exclude_ids = array_values( array_unique( $exclude_ids ) );
			if ( count( $exclude_ids ) > 500 ) {
				$exclude_ids = array_slice( $exclude_ids, 0, 500 );
			}
		}

		if ( $next_id <= 0 ) {
			wp_send_json_success( array( 'done' => true, 'message' => '没有更多可处理文章了' ) );
		}

		$post = get_post( $next_id );
		if ( ! $post || $post->post_type !== 'post' ) {
			delete_post_meta( $next_id, $lock_key );
			wp_send_json_success( array(
				'done' => false,
				'ok' => false,
				'post_id' => $next_id,
				'post_title' => get_the_title( $next_id ),
				'message' => '文章不存在或类型不支持',
			) );
		}
		if ( ! current_user_can( 'edit_post', $next_id ) ) {
			delete_post_meta( $next_id, $lock_key );
			wp_send_json_success( array(
				'done' => false,
				'ok' => false,
				'post_id' => $next_id,
				'post_title' => get_the_title( $next_id ),
				'message' => '无权限编辑该文章',
			) );
		}

		$title = get_the_title( $next_id );
		$title = is_string( $title ) ? trim( $title ) : '';
		if ( $title === '' ) {
			$title = 'Untitled';
		}

		$excerpt_text = '';
		$content = isset( $post->post_content ) ? (string) $post->post_content : '';
		$content = trim( $content );
		if ( $content !== '' ) {
			$excerpt_text = wp_strip_all_tags( $content );
			$excerpt_text = trim( $excerpt_text );
			if ( $excerpt_text !== '' ) {
				if ( function_exists( 'mb_substr' ) ) {
					$excerpt_text = mb_substr( $excerpt_text, 0, 2000 );
				} else {
					$excerpt_text = substr( $excerpt_text, 0, 2000 );
				}
			}
		}

		$image_admin = WP_AIW_Image_Admin::instance();
		$deleted_attachment_id = 0;
		if ( $mode === 'regenerate_all' ) {
			$del = $image_admin->delete_featured_image_and_attachment( $next_id );
			if ( empty( $del['ok'] ) ) {
				delete_post_meta( $next_id, $lock_key );
				wp_send_json_success( array(
					'done' => false,
					'ok' => false,
					'post_id' => $next_id,
					'post_title' => get_the_title( $next_id ),
					'message' => isset( $del['error'] ) ? (string) $del['error'] : '删除原特色图失败',
				) );
			}
			if ( ! empty( $del['attachment_id'] ) ) {
				$deleted_attachment_id = (int) $del['attachment_id'];
			}
		}

		$r = $image_admin->generate_upload_and_set_featured_image( array(
			'post_id' => $next_id,
			'title' => $title,
			'requirements' => $requirements,
			'language' => $language,
			'excerpt_text' => $excerpt_text,
			'style_hint' => $style_hint,
			'size' => $size,
		) );

		delete_post_meta( $next_id, $lock_key );

		if ( empty( $r['ok'] ) ) {
			$message = isset( $r['error'] ) ? (string) $r['error'] : '生成失败';
			wp_send_json_success( array(
				'done' => false,
				'ok' => false,
				'post_id' => $next_id,
				'post_title' => get_the_title( $next_id ),
				'message' => $message,
				'deleted_attachment_id' => $deleted_attachment_id,
				'marked_failed' => 0,
			) );
		}

		wp_send_json_success( array(
			'done' => false,
			'ok' => true,
			'post_id' => $next_id,
			'post_title' => get_the_title( $next_id ),
			'attachment_id' => isset( $r['attachment_id'] ) ? (int) $r['attachment_id'] : 0,
			'deleted_attachment_id' => $deleted_attachment_id,
			'message' => '已生成并设置特色图',
		) );
	}
}
